// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.utils;

import java.util.List;
import java.util.Map;
import java.util.Scanner;

import com.appslandia.common.base.CaseInsensitiveMap;
import com.appslandia.common.base.Config;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class ConsoleApp {

	final String title;
	final Map<String, CommandHandler> handlers = new CaseInsensitiveMap<>();

	public ConsoleApp(String title) {
		this.title = title;
	}

	public void registerCommand(String command, CommandHandler handler) {
		AssertUtils.assertTrue(!this.handlers.containsKey(command), "handler is already added.");
		this.handlers.put(command, handler);
	}

	public void run() {
		Scanner scanner = new Scanner(System.in);
		System.out.println(this.title);

		while (true) {
			System.out.print("> ");
			String input = StringUtils.trimToEmpty(scanner.nextLine());
			if (input.isEmpty()) {
				continue;
			}
			if ("x".equalsIgnoreCase(input)) {
				break;
			}

			List<String> items = SplitUtils.splitByWhitespace(input);
			String command = items.remove(0);
			String[] args = items.toArray(new String[items.size()]);

			CommandHandler handler = this.handlers.get(command);
			if (handler == null) {
				System.out.println(String.format("Command '%s' is required.", command));
				continue;
			}
			try {
				handler.handle(CLIUtils.parseArgs(args));
			} catch (Throwable ex) {
				ex.printStackTrace(System.out);
			}
		}
		scanner.close();
	}

	public interface CommandHandler {
		void handle(Config args) throws Exception;
	}
}
