// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.utils;

import com.appslandia.common.base.Base64;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class BaseEncodingUtils {

	public static final String ENCODING_BASE64 = "base64";
	public static final String ENCODING_URL_BASE64 = "urlBase64";

	public static boolean isSupported(String baseEncoding) {
		return ENCODING_BASE64.equalsIgnoreCase(baseEncoding) || ENCODING_URL_BASE64.equalsIgnoreCase(baseEncoding);
	}

	public static String encode(byte[] src, String baseEncoding) {
		AssertUtils.assertTrue(isSupported(baseEncoding));

		if (ENCODING_BASE64.equalsIgnoreCase(baseEncoding)) {
			return encodeBase64ToString(src);
		}
		return urlEncodeBase64ToString(src);
	}

	public static byte[] decode(String src, String baseEncoding) {
		AssertUtils.assertTrue(isSupported(baseEncoding));

		if (ENCODING_BASE64.equalsIgnoreCase(baseEncoding)) {
			return decodeBase64FromString(src);
		}
		return urlDecodeBase64FromString(src);
	}

	public static String encodeBase64ToString(byte[] src) {
		return new String(Base64.getDelegate().encode(src), CharsetUtils.ISO_8859_1);
	}

	public static byte[] decodeBase64FromString(String src) {
		return Base64.getDelegate().decode(src.getBytes(CharsetUtils.ISO_8859_1));
	}

	public static String encodeBase64(String src) {
		return encodeBase64ToString(src.getBytes(CharsetUtils.UTF_8));
	}

	public static String decodeBase64(String src) {
		return new String(decodeBase64FromString(src), CharsetUtils.UTF_8);
	}

	public static String urlEncodeBase64ToString(byte[] src) {
		return new String(Base64.getDelegate().urlEncode(src), CharsetUtils.ISO_8859_1);
	}

	public static byte[] urlDecodeBase64FromString(String src) {
		return Base64.getDelegate().urlDecode(src.getBytes(CharsetUtils.ISO_8859_1));
	}

	public static String urlEncodeBase64(String src) {
		return urlEncodeBase64ToString(src.getBytes(CharsetUtils.UTF_8));
	}

	public static String urlDecodeBase64(String src) {
		return new String(urlDecodeBase64FromString(src), CharsetUtils.UTF_8);
	}

	private static final char[] HEX_CHARS = CharUtils.toCharRanges("0-9a-f");

	public static void appendAsHex(StringBuilder builder, byte[] src) {
		for (byte b : src) {
			builder.append(HEX_CHARS[(b & 0xf0) >> 4]);
			builder.append(HEX_CHARS[b & 0x0f]);
		}
	}
}
