// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.jpa;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.persistence.CacheRetrieveMode;
import javax.persistence.CacheStoreMode;
import javax.persistence.EntityGraph;
import javax.persistence.PessimisticLockScope;

import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.base.MapAccessor;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class JpaHints extends InitializeObject implements MapAccessor<String, Object> {

	public static final String CACHE_STORE_MODE = "javax.persistence.cache.storeMode";
	public static final String CACHE_RETRIEVE_MODE = "javax.persistence.cache.retrieveMode";

	public static final String JPA_FETCH_GRAPH = "javax.persistence.fetchgraph";
	public static final String JPA_LOAD_GRAPH = "javax.persistence.loadgraph";

	public static final String PESSIMISTIC_LOCK_SCOPE = "javax.persistence.lock.scope";
	public static final String PESSIMISTIC_LOCK_TIMEOUT = "javax.persistence.lock.timeout";

	public static final JpaHints CACHE_STORE_USE = new JpaHints().hint(CACHE_STORE_MODE, CacheStoreMode.USE).initialize();
	public static final JpaHints CACHE_STORE_REFRESH = new JpaHints().hint(CACHE_STORE_MODE, CacheStoreMode.REFRESH).initialize();
	public static final JpaHints CACHE_STORE_BYPASS = new JpaHints().hint(CACHE_STORE_MODE, CacheStoreMode.BYPASS).initialize();

	public static final JpaHints CACHE_RETRIEVE_USE = new JpaHints().hint(CACHE_RETRIEVE_MODE, CacheRetrieveMode.USE).initialize();
	public static final JpaHints CACHE_RETRIEVE_BYPASS = new JpaHints().hint(CACHE_RETRIEVE_MODE, CacheRetrieveMode.BYPASS).initialize();

	public static final JpaHints PESSIMISTIC_LOCK_EXTENDED = new JpaHints().hint(PESSIMISTIC_LOCK_SCOPE, PessimisticLockScope.EXTENDED).initialize();

	final Map<String, Object> hints = new HashMap<>();

	@Override
	protected void init() throws Exception {
	}

	public JpaHints hint(String key, Object value) {
		assertNotInitialized();
		this.hints.put(key, value);
		return this;
	}

	public JpaHints fetchgraph(EntityGraph<?> graph) {
		assertNotInitialized();
		this.hints.put(JPA_FETCH_GRAPH, graph);
		return this;
	}

	public JpaHints loadgraph(EntityGraph<?> graph) {
		assertNotInitialized();
		this.hints.put(JPA_LOAD_GRAPH, graph);
		return this;
	}

	@Override
	public int size() {
		initialize();
		return this.hints.size();
	}

	@Override
	public boolean isEmpty() {
		initialize();
		return this.hints.isEmpty();
	}

	@Override
	public boolean containsKey(Object key) {
		initialize();
		return this.hints.containsKey(key);
	}

	@Override
	public Object get(Object key) {
		initialize();
		return this.hints.get(key);
	}

	@Override
	public Set<String> keySet() {
		initialize();
		return this.hints.keySet();
	}

	@Override
	public Set<java.util.Map.Entry<String, Object>> entrySet() {
		initialize();
		return this.hints.entrySet();
	}
}
