// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.crypto;

import com.appslandia.common.base.DestroyException;
import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.utils.ArrayUtils;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.MathUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class SignedEncryptor extends InitializeObject implements Encryptor {

	private Encryptor encryptor;
	private Digester signer;

	public SignedEncryptor() {
	}

	public SignedEncryptor(Encryptor encryptor, Digester signer) {
		this.encryptor = encryptor;
		this.signer = signer;
	}

	@Override
	protected void init() throws Exception {
		AssertUtils.assertNotNull(this.encryptor, "encryptor is required.");
		AssertUtils.assertNotNull(this.signer, "signer is required.");
	}

	@Override
	public void destroy() throws DestroyException {
		if (this.encryptor != null) {
			this.encryptor.destroy();
		}
		if (this.signer != null) {
			this.signer.destroy();
		}
		super.destroy();
	}

	@Override
	public byte[] encrypt(byte[] message) throws CryptoException {
		this.initialize();
		AssertUtils.assertNotNull(message, "message is required.");

		byte[] encrypted = this.encryptor.encrypt(message);
		byte[] digested = this.signer.digest(encrypted);

		if (this.signer.getDigestLength() > 0) {
			return ArrayUtils.append(digested, encrypted);

		} else {
			byte[] len = MathUtils.toByteArray(digested.length);
			return ArrayUtils.append(len, digested, encrypted);
		}
	}

	@Override
	public byte[] decrypt(byte[] message) throws CryptoException {
		this.initialize();
		AssertUtils.assertNotNull(message, "message is required.");

		byte[] digested = null;
		byte[] parsedMsg = null;

		int digestLength = this.signer.getDigestLength();
		if (digestLength > 0) {
			digested = new byte[digestLength];

			AssertUtils.assertTrue(message.length > digestLength, "message is invalid.");
			parsedMsg = new byte[message.length - digestLength];
			ArrayUtils.copy(message, digested, parsedMsg);

		} else {
			byte[] len = new byte[4];
			AssertUtils.assertTrue(message.length > 4, "message is invalid.");
			ArrayUtils.copy(message, len);

			digestLength = MathUtils.toInt(len);
			AssertUtils.assertTrue(digestLength > 0, "message is invalid.");

			digested = new byte[digestLength];
			parsedMsg = new byte[message.length - digestLength - 4];
			System.arraycopy(message, 4, digested, 0, digestLength);
			System.arraycopy(message, 4 + digestLength, parsedMsg, 0, parsedMsg.length);
		}

		if (!this.signer.verify(parsedMsg, digested)) {
			throw new CryptoException("message was tampered.");
		}
		return this.encryptor.decrypt(parsedMsg);
	}

	public SignedEncryptor setEncryptor(Encryptor encryptor) {
		assertNotInitialized();
		this.encryptor = encryptor;
		return this;
	}

	public SignedEncryptor setSigner(Digester signer) {
		assertNotInitialized();
		this.signer = signer;
		return this;
	}

	@Override
	public SignedEncryptor copy() {
		SignedEncryptor impl = new SignedEncryptor();
		if (this.encryptor != null) {
			impl.encryptor = this.encryptor.copy();
		}
		if (this.signer != null) {
			impl.signer = this.signer.copy();
		}
		return impl;
	}
}
