// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.crypto;

import java.security.SecureRandom;
import java.util.Random;
import java.util.regex.Pattern;

import com.appslandia.common.base.Permutation;
import com.appslandia.common.base.WordsGenerator;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.CharUtils;
import com.appslandia.common.utils.RandomUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class PasswordGenerator {

	// must contain one digit from 0-9
	// must contain one lower case characters
	// must contain one upper case characters
	// must contain one special symbols in the list $@&#!?*%:+-
	// length 8-32
	public static final Pattern PASSWORD_PATTERN = Pattern.compile("(?=.*\\d)(?=.*[a-z])(?=.*[A-Z])(?=.*[$@&#!?*%:+-]).{8,32}");
	private static final char[] SYMBOLS = "$@&#!?*%:+-".toCharArray();

	private static final char[] UPPER_CASES = CharUtils.toCharRanges("A-Z");
	private static final char[] LOWER_CASES = CharUtils.toCharRanges("a-z");
	private static final char[] DIGITS = CharUtils.toCharRanges("0-9");

	final Random random = new SecureRandom();

	public char[] generatePassword(int minLength, int maxLength) {
		AssertUtils.assertTrue(minLength <= maxLength, "minLength <= maxLength");
		AssertUtils.assertTrue(minLength >= 8, "minLength >= 8");
		AssertUtils.assertTrue(maxLength <= 32, "maxLength <= 32");

		int length = RandomUtils.nextInt(this.random, minLength, maxLength);
		char[] pwdChars = new char[length];

		int emptyCount = length;
		Permutation sourceIndexes = new Permutation(4);
		while (sourceIndexes.hasNext()) {

			switch (sourceIndexes.next()) {
			case 0:
				emptyCount = WordsGenerator.randomChars(this.random, UPPER_CASES, pwdChars, 1, emptyCount);
				break;
			case 1:
				emptyCount = WordsGenerator.randomChars(this.random, LOWER_CASES, pwdChars, 1, emptyCount);
				break;
			case 2:
				emptyCount = WordsGenerator.randomChars(this.random, DIGITS, pwdChars, 1, emptyCount);
				break;
			default:
				emptyCount = WordsGenerator.randomChars(this.random, SYMBOLS, pwdChars, 1, emptyCount);
				break;
			}
		}

		for (int i = 0; i < length; i++) {
			if (pwdChars[i] == 0) {
				int src = this.random.nextInt(4);
				switch (src) {
				case 0:
					pwdChars[i] = UPPER_CASES[this.random.nextInt(UPPER_CASES.length)];
					break;
				case 1:
					pwdChars[i] = LOWER_CASES[this.random.nextInt(LOWER_CASES.length)];
					break;
				case 2:
					pwdChars[i] = DIGITS[this.random.nextInt(DIGITS.length)];
					break;
				default:
					pwdChars[i] = SYMBOLS[this.random.nextInt(SYMBOLS.length)];
					break;
				}
			}
		}
		return pwdChars;
	}

	public boolean verifyPassword(String password) {
		AssertUtils.assertNotNull(password);
		return PASSWORD_PATTERN.matcher(password).matches();
	}
}
