// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.cdi;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Set;

import javax.enterprise.event.Observes;
import javax.enterprise.inject.spi.Extension;
import javax.enterprise.inject.spi.ProcessAnnotatedType;

import com.appslandia.common.base.DeployPhase;
import com.appslandia.common.base.EnablePhases;
import com.appslandia.common.utils.CollectionUtils;
import com.appslandia.common.utils.ObjectUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class CDIVetoExtension implements Extension {

	final Set<Class<?>> vetoedClasses = new HashSet<>();
	final Set<String> vetoedPackages = new HashSet<>();
	final Set<Class<? extends Annotation>> vetoedAnnotations = new HashSet<>();

	protected void vetoClasses(Class<?>... beanClasses) {
		CollectionUtils.add(this.vetoedClasses, beanClasses);
	}

	protected void vetoPackages(String... packages) {
		CollectionUtils.add(this.vetoedPackages, packages);
	}

	protected void vetoPackages(Class<?>... beanClasses) {
		for (Class<?> clazz : beanClasses) {
			this.vetoedPackages.add(clazz.getPackage().getName());
		}
	}

	// vetoAnnotations(Class<? extends Annotation>... annotationClasses)
	protected void vetoAnnotations(Class<?>... annotationClasses) {
		for (Class<?> clazz : annotationClasses) {
			this.vetoedAnnotations.add(ObjectUtils.cast(clazz));
		}
	}

	protected boolean willVetoClass(Class<?> beanClass) {
		return false;
	}

	protected void onVetoClass(Class<?> beanClass) {
	}

	protected void onDeployClass(Class<?> beanClass) {
	}

	public <T> void processAnotated(@Observes ProcessAnnotatedType<T> event) {
		Class<?> beanClass = event.getAnnotatedType().getJavaClass();

		if (willVetoClasses(this.vetoedClasses, beanClass)) {
			event.veto();
			onVetoClass(beanClass);
			return;
		}
		if (willVetoPackages(this.vetoedPackages, beanClass)) {
			event.veto();
			onVetoClass(beanClass);
			return;
		}
		if (willVetoAnnotations(this.vetoedAnnotations, beanClass)) {
			event.veto();
			onVetoClass(beanClass);
			return;
		}

		// @EnablePhases
		EnablePhases enablePhases = getEnablePhases(beanClass);
		if ((enablePhases != null) && !DeployPhase.getCurrent().isAny(enablePhases.value())) {
			event.veto();
			onVetoClass(beanClass);
			return;
		}

		// Others
		if (willVetoClass(beanClass)) {
			event.veto();
			onVetoClass(beanClass);
			return;
		}
		onDeployClass(beanClass);
	}

	public static boolean willVetoClasses(Set<Class<?>> vetoedClasses, Class<?> beanClass) {
		if (vetoedClasses.isEmpty()) {
			return false;
		}
		if (vetoedClasses.contains(beanClass)) {
			return true;
		}

		// CDIFactory
		if (CDIFactory.class.isAssignableFrom(beanClass)) {
			Method factoryMethod = CDIUtils.getFactoryMethod(ObjectUtils.cast(beanClass));
			if (vetoedClasses.contains(factoryMethod.getReturnType())) {
				return true;
			}
		}
		return false;
	}

	public static boolean willVetoPackages(Set<String> vetoedPackages, Class<?> beanClass) {
		if (vetoedPackages.isEmpty()) {
			return false;
		}
		for (String pkg : vetoedPackages) {
			if (beanClass.getPackage().getName().equals(pkg)) {
				return true;
			}
		}

		// CDIFactory
		if (CDIFactory.class.isAssignableFrom(beanClass)) {
			Method factoryMethod = CDIUtils.getFactoryMethod(ObjectUtils.cast(beanClass));
			for (String pkg : vetoedPackages) {
				if (factoryMethod.getReturnType().getPackage().getName().equals(pkg)) {
					return true;
				}
			}
		}
		return false;
	}

	public static boolean willVetoAnnotations(Set<Class<? extends Annotation>> vetoedAnnotations, Class<?> beanClass) {
		if (vetoedAnnotations.isEmpty()) {
			return false;
		}
		for (Class<? extends Annotation> annotationClass : vetoedAnnotations) {
			if (beanClass.getDeclaredAnnotation(annotationClass) != null) {
				return true;
			}
		}

		// CDIFactory
		if (CDIFactory.class.isAssignableFrom(beanClass)) {
			Method factoryMethod = CDIUtils.getFactoryMethod(ObjectUtils.cast(beanClass));
			for (Class<? extends Annotation> annotationClass : vetoedAnnotations) {
				if (factoryMethod.getDeclaredAnnotation(annotationClass) != null) {
					return true;
				}
			}
		}
		return false;
	}

	public static EnablePhases getEnablePhases(Class<?> beanClass) {
		EnablePhases enablePhases = beanClass.getDeclaredAnnotation(EnablePhases.class);
		if (enablePhases != null) {
			return enablePhases;
		}

		// CDIFactory
		if (CDIFactory.class.isAssignableFrom(beanClass)) {
			Method factoryMethod = CDIUtils.getFactoryMethod(ObjectUtils.cast(beanClass));
			return factoryMethod.getDeclaredAnnotation(EnablePhases.class);
		}
		return null;
	}
}
