// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.base;

import java.util.Locale;

import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.ObjectUtils;
import com.appslandia.common.utils.ReflectionUtils;
import com.appslandia.common.utils.StringUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class Language extends InitializeObject {

	public static final String TIME_FORMAT_24H = "HH:mm";
	public static final String TIME_FORMAT_24H_Z = "HH:mmXXX";

	public static final String TIME_FORMAT = "HH:mm:ss.SSS";
	public static final String TIME_FORMAT_Z = "HH:mm:ss.SSSXXX";

	public static final Language EN = new Language().setLocale(Locale.US).setDateFormat("MM/dd/yyyy").initialize();
	public static final Language VI = new Language().setLocale(new Locale("vi", "VN")).setDateFormat("dd/MM/yyyy").initialize();

	private Locale locale;
	private String dateFormat;

	private String dateTimeFormat;
	private String dateTimeFormatZ;

	private String dateTimeFormat24H;
	private String dateTimeFormat24HZ;

	@Override
	protected void init() throws Exception {
		AssertUtils.assertNotNull(this.locale, "locale is required.");
		AssertUtils.assertNotNull(this.dateFormat, "dateFormat is required.");

		this.dateTimeFormat = String.format("%s %s", this.dateFormat, TIME_FORMAT);
		this.dateTimeFormatZ = String.format("%s %s", this.dateFormat, TIME_FORMAT_Z);

		this.dateTimeFormat24H = String.format("%s %s", this.dateFormat, TIME_FORMAT_24H);
		this.dateTimeFormat24HZ = String.format("%s %s", this.dateFormat, TIME_FORMAT_24H_Z);
	}

	public String getId() {
		this.initialize();
		return this.locale.getLanguage();
	}

	public String getDisplayLanguage() {
		this.initialize();
		return this.locale.getDisplayLanguage(this.locale);
	}

	public Locale getLocale() {
		this.initialize();
		return this.locale;
	}

	public Language setLocale(Locale locale) {
		this.assertNotInitialized();
		this.locale = locale;
		return this;
	}

	public String getDateFormat() {
		this.initialize();
		return this.dateFormat;
	}

	public Language setDateFormat(String dateFormat) {
		this.assertNotInitialized();
		this.dateFormat = StringUtils.trimToNull(dateFormat);
		return this;
	}

	public String getDateTimeFormat() {
		this.initialize();
		return this.dateTimeFormat;
	}

	public String getDateTimeFormatZ() {
		this.initialize();
		return this.dateTimeFormatZ;
	}

	public String getDateTimeFormat24H() {
		this.initialize();
		return this.dateTimeFormat24H;
	}

	public String getDateTimeFormat24HZ() {
		this.initialize();
		return this.dateTimeFormat24HZ;
	}

	public String getTimeFormat24H() {
		this.initialize();
		return TIME_FORMAT_24H;
	}

	public String getTimeFormat24HZ() {
		this.initialize();
		return TIME_FORMAT_24H_Z;
	}

	public String getTimeFormat() {
		this.initialize();
		return TIME_FORMAT;
	}

	public String getTimeFormatZ() {
		this.initialize();
		return TIME_FORMAT_Z;
	}

	@Override
	public int hashCode() {
		return ObjectUtils.hashCode(this.getId().toLowerCase(Locale.ENGLISH));
	}

	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof Language)) {
			return false;
		}
		Language another = (Language) obj;
		return this.getId().equalsIgnoreCase(another.getId());
	}

	private static volatile Language current;
	private static final Object MUTEX = new Object();

	public static Language getCurrent() {
		Language obj = current;
		if (obj == null) {
			synchronized (MUTEX) {
				if ((obj = current) == null) {
					current = obj = initLanguage();
				}
			}
		}
		return obj;
	}

	public static void setCurrent(Language obj) {
		AssertUtils.assertNull(current);
		current = obj;
	}

	public static final String SYSTEM_PROP_LANGUAGE = "__language";

	private static Language initLanguage() {
		try {
			String implName = System.getProperty(SYSTEM_PROP_LANGUAGE);
			if (implName == null) {
				return EN;
			}
			Class<? extends Language> implClass = ReflectionUtils.loadClass(implName, null);
			return implClass.newInstance();
		} catch (Exception ex) {
			throw new InitializeException(ex);
		}
	}
}
