// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.base;

import com.appslandia.common.utils.AssertUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public enum DeployPhase {
	DEVELOPMENT, TESTING, STAGING, PRODUCTION;

	public boolean isDevelopment() {
		return this == DEVELOPMENT;
	}

	public boolean isTesting() {
		return this == TESTING;
	}

	public boolean isStaging() {
		return this == STAGING;
	}

	public boolean isProduction() {
		return this == PRODUCTION;
	}

	public boolean isNotProduction() {
		return this != PRODUCTION;
	}

	public boolean isAny(DeployPhase... phases) {
		for (DeployPhase phase : phases) {
			if (this == phase) {
				return true;
			}
		}
		return false;
	}

	private static volatile DeployPhase current;
	private static final Object MUTEX = new Object();

	public static DeployPhase getCurrent() {
		DeployPhase obj = current;
		if (obj == null) {
			synchronized (MUTEX) {
				if ((obj = current) == null) {
					current = obj = initDeployPhase();
				}
			}
		}
		return obj;
	}

	public static void setCurrent(DeployPhase phrase) {
		AssertUtils.assertNull(current);
		current = phrase;
	}

	public static final String SYSTEM_PROP_DEPLOY_PHASE = "__deployPhase";

	private static DeployPhase initDeployPhase() {
		try {
			String phase = System.getProperty(SYSTEM_PROP_DEPLOY_PHASE, DeployPhase.DEVELOPMENT.name());
			return DeployPhase.valueOf(phase);
		} catch (Exception ex) {
			throw new InitializeException(ex);
		}
	}
}
