// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.utils;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class ReflectionUtils {

	public static final Object[] EMPTY_OBJECTS = {};
	public static final Class<?>[] EMPTY_CLASSES = {};
	public static final Annotation[] EMPTY_ANNOTATIONS = {};

	public static interface FieldMatcher {
		boolean match(Field field);
	}

	public static interface FieldTraverser {
		boolean apply(Field field) throws ReflectionException;
	}

	public static interface MethodMatcher {
		boolean match(Method m);
	}

	public static interface MethodTraverser {
		boolean apply(Method m) throws ReflectionException;
	}

	public static final FieldTraverser FIND_FIELD_TRAVERSER = new FieldTraverser() {

		@Override
		public boolean apply(Field field) throws ReflectionException {
			return true;
		}
	};

	public static final MethodTraverser FIND_METHOD_TRAVERSER = new MethodTraverser() {

		@Override
		public boolean apply(Method m) throws ReflectionException {
			return true;
		}
	};

	public static Field findField(Class<?> clazz, final String property) throws ReflectionException {
		return traverse(clazz, new FieldMatcher() {

			@Override
			public boolean match(Field f) {
				if (f.getName().equals(property)) {
					return true;
				}
				if (f.getType() == boolean.class) {
					if (f.getName().equals("is" + StringUtils.firstUpperCase(property))) {
						return true;
					}
				}
				return false;
			}
		}, FIND_FIELD_TRAVERSER);
	}

	public static Method findMethod(Class<?> clazz, final String methodName) throws ReflectionException {
		return traverse(clazz, new MethodMatcher() {

			@Override
			public boolean match(Method m) {
				return m.getName().equals(methodName);
			}
		}, FIND_METHOD_TRAVERSER);
	}

	public static Field traverse(Class<?> clazz, FieldMatcher matcher, FieldTraverser traverser) throws ReflectionException {
		Field last = null;
		while (clazz != null) {
			for (Field field : clazz.getDeclaredFields()) {
				if (matcher.match(field)) {
					last = field;
					if (traverser.apply(field)) {
						return last;
					}
				}
			}
			clazz = clazz.getSuperclass();
		}
		return last;
	}

	public static Method traverse(Class<?> clazz, MethodMatcher matcher, MethodTraverser traverser) throws ReflectionException {
		Method last = null;
		while (clazz != null) {
			for (Method m : clazz.getDeclaredMethods()) {
				if (matcher.match(m)) {
					last = m;
					if (traverser.apply(m)) {
						return last;
					}
				}
			}
			clazz = clazz.getSuperclass();
		}
		return last;
	}

	public static Object invoke(Method m, Object obj, Object... args) throws ReflectionException {
		try {
			return m.invoke(obj, args);
		} catch (IllegalAccessException ex) {
			throw new ReflectionException(ex);

		} catch (InvocationTargetException ex) {
			throw new ReflectionException(ex);
		}
	}

	public static void set(Field m, Object obj, Object value) throws ReflectionException {
		try {
			m.set(obj, value);
		} catch (IllegalAccessException ex) {
			throw new ReflectionException(ex);
		}
	}

	public static Object get(Field m, Object obj) throws ReflectionException {
		try {
			return m.get(obj);
		} catch (IllegalAccessException ex) {
			throw new ReflectionException(ex);
		}
	}

	public static <T> Constructor<T> getConstructor(Class<T> clazz) {
		return getConstructor(clazz, EMPTY_CLASSES);
	}

	public static <T> Constructor<T> getConstructor(Class<T> clazz, Class<?>... paramTypes) {
		try {
			return clazz.getConstructor(paramTypes);
		} catch (NoSuchMethodException ex) {
			return null;
		}
	}

	public static boolean isPublicStaticFinal(int modifier) {
		return Modifier.isPublic(modifier) && Modifier.isStatic(modifier) && Modifier.isFinal(modifier);
	}
}
