// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.utils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.EnumMap;
import java.util.GregorianCalendar;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import com.appslandia.common.base.DateFormatException;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class DateUtils {

	// ISO8601 Date Formats
	public static final String DATE_FORMAT = "yyyy-MM-dd";
	public static final String TIME_FORMAT = "HH:mm:ss.SSS";
	public static final String DATETIME_FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSS";

	public static final String TIME_FORMAT_Z = "HH:mm:ss.SSSXXX";
	public static final String DATETIME_FORMAT_Z = "yyyy-MM-dd'T'HH:mm:ss.SSSXXX";

	public static final long __1SEC_MS = 1000;
	public static final long __1MIN_MS = __1SEC_MS * 60;
	public static final long __1HR_MS = __1MIN_MS * 60;
	public static final long __1D_MS = __1HR_MS * 24;
	public static final long __1WK_MS = __1D_MS * 7;

	private static final Pattern TEMPORAL_AMT_PATTERN = Pattern.compile("(\\d+(wk|d|hr|min|sec|ms)\\s*)+", Pattern.CASE_INSENSITIVE);

	public static long translateToMs(String temporalAmt) throws IllegalArgumentException {
		temporalAmt = StringUtils.trimToNull(temporalAmt);
		AssertUtils.assertNotNull(temporalAmt, "temporalAmt is required.");
		AssertUtils.assertTrue(TEMPORAL_AMT_PATTERN.matcher(temporalAmt).matches(), "temporalAmt is invalid format.");

		long result = 0l;
		int i = 0;

		while (i < temporalAmt.length()) {
			int j = i;
			while (Character.isDigit(temporalAmt.charAt(j)))
				j++;
			int k = j;
			while ((k <= temporalAmt.length() - 1) && (Character.isLetter(temporalAmt.charAt(k)) || (temporalAmt.charAt(k) == (' '))))
				k++;

			int amt = Integer.parseInt(temporalAmt.substring(i, j));
			String unit = temporalAmt.substring(j, k).trim();

			if ("wk".equalsIgnoreCase(unit)) {
				result += __1WK_MS * amt;
			} else if ("d".equalsIgnoreCase(unit)) {
				result += __1D_MS * amt;
			} else if ("hr".equalsIgnoreCase(unit)) {
				result += __1HR_MS * amt;
			} else if ("min".equalsIgnoreCase(unit)) {
				result += __1MIN_MS * amt;
			} else if ("sec".equalsIgnoreCase(unit)) {
				result += __1SEC_MS * amt;
			} else {
				result += amt;
			}

			i = k;
		}
		return result;
	}

	public static Map<TimeUnit, Long> parseUnits(long durationMs, TimeUnit start, TimeUnit end) {
		return parseUnits(durationMs, TimeUnit.MILLISECONDS, start, end);
	}

	public static Map<TimeUnit, Long> parseUnits(long duration, TimeUnit unit, TimeUnit start, TimeUnit end) {
		AssertUtils.assertTrue(start.compareTo(end) >= 0);
		Map<TimeUnit, Long> map = new EnumMap<>(TimeUnit.class);

		long ms = TimeUnit.MILLISECONDS.convert(duration, unit);
		TimeUnit u = start;
		while (true) {
			long v = u.convert(ms, TimeUnit.MILLISECONDS);
			map.put(u, v);
			if (u == end) {
				break;
			}
			ms -= TimeUnit.MILLISECONDS.convert(v, u);
			u = nextUnit(u);
		}
		return map;
	}

	private static TimeUnit nextUnit(TimeUnit unit) {
		switch (unit) {
		case DAYS:
			return TimeUnit.HOURS;
		case HOURS:
			return TimeUnit.MINUTES;
		case MINUTES:
			return TimeUnit.SECONDS;
		case SECONDS:
			return TimeUnit.MILLISECONDS;
		default:
			throw new IllegalArgumentException();
		}
	}

	public static long timeMillisAt(int days) {
		return System.currentTimeMillis() + (days * __1D_MS);
	}

	public static java.sql.Date today() {
		return new java.sql.Date(todayCalendar().getTimeInMillis());
	}

	public static java.sql.Timestamp now() {
		return new java.sql.Timestamp(System.currentTimeMillis());
	}

	public static Calendar todayCalendar() {
		Calendar cal = new GregorianCalendar();
		clearTime(cal);
		return cal;
	}

	public static Calendar nowCalendar() {
		return new GregorianCalendar();
	}

	public static Calendar getCalendar(int dayOfWeek, int atHour, int atMinute) {
		Calendar cal = new GregorianCalendar();

		cal.set(Calendar.DAY_OF_WEEK, dayOfWeek);
		cal.set(Calendar.HOUR_OF_DAY, atHour);
		cal.set(Calendar.MINUTE, atMinute);

		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		return cal;
	}

	public static java.util.Date copyTime(java.util.Date toDate, java.util.Date fromDate) {
		Calendar toCal = new GregorianCalendar();
		toCal.setTime(toDate);

		Calendar fromCal = new GregorianCalendar();
		fromCal.setTime(fromDate);

		toCal.set(Calendar.HOUR_OF_DAY, fromCal.get(Calendar.HOUR_OF_DAY));
		toCal.set(Calendar.MINUTE, fromCal.get(Calendar.MINUTE));
		toCal.set(Calendar.SECOND, fromCal.get(Calendar.SECOND));
		toCal.set(Calendar.MILLISECOND, fromCal.get(Calendar.MILLISECOND));

		return toCal.getTime();
	}

	public static java.util.Date clearTime(java.util.Date d) {
		Calendar cal = new GregorianCalendar();
		cal.setTime(d);

		clearTime(cal);
		return cal.getTime();
	}

	public static void clearTime(Calendar cal) {
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
	}

	public static long clearMillis(long timeInMillis) {
		Calendar cal = new GregorianCalendar();
		cal.setTimeInMillis(timeInMillis);
		cal.set(Calendar.MILLISECOND, 0);
		return cal.getTimeInMillis();
	}

	public static String format(java.util.Date temporal, String pattern) {
		return newDateFormat(pattern).format(temporal);
	}

	public static SimpleDateFormat newDateFormat(String pattern) {
		SimpleDateFormat sdf = new SimpleDateFormat(pattern);
		sdf.setLenient(false);
		return sdf;
	}

	public static java.sql.Date iso8601Date(String date) throws DateFormatException {
		return (date != null) ? new java.sql.Date(parse(date, DATE_FORMAT).getTime()) : null;
	}

	public static String iso8601Date(java.util.Date date) {
		return (date != null) ? newDateFormat(DATE_FORMAT).format(date) : null;
	}

	public static java.sql.Time iso8601Time(String time) throws DateFormatException {
		return (time != null) ? new java.sql.Time(parse(time, TIME_FORMAT).getTime()) : null;
	}

	public static String iso8601Time(java.util.Date time) {
		return (time != null) ? newDateFormat(TIME_FORMAT).format(time) : null;
	}

	public static java.sql.Timestamp iso8601DateTime(String dateTime) throws DateFormatException {
		return (dateTime != null) ? new java.sql.Timestamp(parse(dateTime, DATETIME_FORMAT).getTime()) : null;
	}

	public static String iso8601DateTime(java.util.Date dateTime) {
		return (dateTime != null) ? newDateFormat(DATETIME_FORMAT).format(dateTime) : null;
	}

	private static java.util.Date parse(String temporal, String pattern) throws DateFormatException {
		try {
			return newDateFormat(pattern).parse(temporal);
		} catch (ParseException ex) {
			throw new DateFormatException(ex);
		}
	}
}
