// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.objects;

import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Parameter;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;

import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.ObjectUtils;
import com.appslandia.common.utils.ReflectionUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class ObjectFactory extends InitializeObject {

	protected final Set<ObjectInst> objectInsts = new LinkedHashSet<>();
	protected final ConcurrentMap<KeyDesc, ObjectInst> objectInstMap = new ConcurrentHashMap<>();

	@Override
	protected void init() throws Exception {
		validateFactory();
	}

	protected void validateFactory() throws ObjectException {
		for (ObjectInst objInst : this.objectInsts) {
			if (objInst.getProducer() != null) {
				continue;
			}
			new InjectTraverser() {

				@Override
				public boolean isValidating() {
					return true;
				}

				@Override
				public void onParameter(Parameter parameter) throws ObjectException {
					validateInject(parameter.getType(), AnnotationUtils.parseQualifiers(parameter), parameter);
				}

				@Override
				public void onField(Field field) throws ObjectException {
					validateInject(field.getType(), AnnotationUtils.parseQualifiers(field), field);
				}

				@Override
				public void onMethod(Method method) throws ObjectException {
					throw new UnsupportedOperationException();
				}

				private String toString(AnnotatedElement element) {
					if (element instanceof Parameter) {
						return ((Parameter) element).getDeclaringExecutable().toString();
					}
					return element.toString();
				}

				private void validateInject(Class<?> type, Annotation[] qualifiers, AnnotatedElement element) throws ObjectException {
					if (ObjectFactory.class.isAssignableFrom(type)) {
						return;
					}
					int count = getMatchingCount(type, qualifiers);
					if (count == 0) {
						throw new ObjectException("Unsatisfied dependency (type=" + type + ", qualifiers=" + Arrays.toString(qualifiers) + ", location=" + toString(element) + ")");
					}
					if (count > 1) {
						throw new ObjectException("Ambiguous dependency (type=" + type + ", qualifiers=" + Arrays.toString(qualifiers) + ", location=" + toString(element) + ")");
					}
				}

			}.traverse(objInst.getType());
		}
	}

	protected int getMatchingCount(Class<?> type, Annotation[] qualifiers) {
		int count = 0;
		for (ObjectInst objInst : this.objectInsts) {
			if (type.isAssignableFrom(objInst.getType()) && AnnotationUtils.equals(qualifiers, objInst.getQualifiers())) {
				count++;
			}
		}
		return count;
	}

	public <T> ObjectFactory register(Class<T> type, ObjectProducer<T> producer) {
		register(type, producer, ObjectScope.SINGLETON, ReflectionUtils.EMPTY_ANNOTATIONS);
		return this;
	}

	public <T> ObjectFactory register(Class<T> type, ObjectProducer<T> producer, ObjectScope scope) {
		return register(type, producer, scope, ReflectionUtils.EMPTY_ANNOTATIONS);
	}

	public <T> ObjectFactory register(Class<T> type, ObjectProducer<T> producer, ObjectScope scope, Annotation... qualifiers) {
		assertNotInitialized();
		AssertUtils.assertNotNull(type);
		AssertUtils.assertNotNull(producer);
		AssertUtils.assertNotNull(scope);

		ObjectInst objInst = new ObjectInst().setType(type).setQualifiers(qualifiers);
		objInst.setScope(scope);
		objInst.setProducer(producer);

		this.objectInsts.add(objInst);
		return this;
	}

	public ObjectFactory register(Class<?> implClass) {
		return register(implClass, ObjectScope.SINGLETON);
	}

	public ObjectFactory register(Class<?> implClass, ObjectScope scope) {
		assertNotInitialized();
		AssertUtils.assertNotNull(implClass);
		AssertUtils.assertNotNull(scope);
		AssertUtils.assertTrue(!Modifier.isAbstract(implClass.getModifiers()));

		ObjectInst objInst = new ObjectInst().setType(implClass).setQualifiers(AnnotationUtils.parseQualifiers(implClass));
		objInst.setScope(scope);

		this.objectInsts.add(objInst);
		return this;
	}

	public ObjectFactory register(Class<?> implClass, ObjectScope scope, Annotation... qualifiers) {
		assertNotInitialized();
		AssertUtils.assertNotNull(implClass);
		AssertUtils.assertNotNull(scope);
		AssertUtils.assertTrue(!Modifier.isAbstract(implClass.getModifiers()));

		ObjectInst objInst = new ObjectInst().setType(implClass).setQualifiers(qualifiers);
		objInst.setScope(scope);

		this.objectInsts.add(objInst);
		return this;
	}

	public ObjectFactory unregister(Class<?> implClass) {
		assertNotInitialized();
		AssertUtils.assertNotNull(implClass);
		this.objectInsts.remove(new ObjectInst().setType(implClass).setQualifiers(AnnotationUtils.parseQualifiers(implClass)));
		return this;
	}

	public ObjectFactory inject(final Object obj) throws ObjectException {
		AssertUtils.assertNotNull(obj);

		new InjectTraverser() {

			@Override
			public boolean isValidating() {
				return false;
			}

			@Override
			public void onParameter(Parameter parameter) throws ObjectException {
				throw new UnsupportedOperationException();
			}

			@Override
			public void onField(Field field) throws ObjectException {
				try {
					field.setAccessible(true);
					Object value = getObject(field.getType(), AnnotationUtils.parseQualifiers(field));
					field.set(obj, value);
				} catch (ObjectException ex) {
					throw ex;
				} catch (Exception ex) {
					throw new ObjectException(ex);
				}
			}

			@Override
			public void onMethod(Method method) throws ObjectException {
				try {
					method.setAccessible(true);
					method.invoke(obj, createArguments(method.getParameters()));
				} catch (ObjectException ex) {
					throw ex;
				} catch (Exception ex) {
					throw new ObjectException(ex);
				}
			}
		}.traverse(obj.getClass());
		return this;
	}

	protected Object[] createArguments(Parameter[] parameters) throws ObjectException {
		Object[] args = new Object[parameters.length];
		for (int i = 0; i < parameters.length; i++) {
			Parameter parameter = parameters[i];
			args[i] = getObject(parameter.getType(), AnnotationUtils.parseQualifiers(parameter));
		}
		return args;
	}

	protected Object produceObject(ObjectInst objInst) throws ObjectException {
		try {
			if (objInst.getProducer() != null) {
				return objInst.getProducer().produce(this);
			}

			Constructor<?> emptyCtor = null, injectCtor = null;
			for (Constructor<?> ctor : objInst.getType().getDeclaredConstructors()) {
				if (ctor.getDeclaredAnnotation(Inject.class) != null) {
					injectCtor = ctor;
					break;
				}
				if (ctor.getParameterCount() == 0) {
					emptyCtor = ctor;
				}
			}
			if ((injectCtor == null) && (emptyCtor == null)) {
				throw new ObjectException("Could not instantiate (implClass=" + objInst.getType() + ")");
			}
			Object instance = null;
			if (injectCtor != null) {
				injectCtor.setAccessible(true);
				instance = injectCtor.newInstance(createArguments(injectCtor.getParameters()));
			} else {
				emptyCtor.setAccessible(true);
				instance = emptyCtor.newInstance(ReflectionUtils.EMPTY_OBJECTS);
			}

			return this.inject(instance).postConstruct(instance);
		} catch (ObjectException ex) {
			throw ex;
		} catch (Exception ex) {
			throw new ObjectException(ex);
		}
	}

	public <T> T getObject(Class<?> type) throws ObjectException {
		return getObject(type, ReflectionUtils.EMPTY_ANNOTATIONS);
	}

	protected ObjectInst getObjectInst(Class<?> type, Annotation[] qualifiers) {
		return this.objectInstMap.computeIfAbsent(new KeyDesc(type, qualifiers), key -> {
			for (ObjectInst objInst : this.objectInsts) {
				if (key.getType().isAssignableFrom(objInst.getType()) && AnnotationUtils.equals(key.getQualifiers(), objInst.getQualifiers())) {
					return objInst;
				}
			}
			return null;
		});
	}

	public <T> T getObject(Class<?> type, Annotation... qualifiers) throws ObjectException {
		AssertUtils.assertNotNull(type);
		initialize();

		// ObjectFactory?
		if (ObjectFactory.class.isAssignableFrom(type)) {
			return ObjectUtils.cast(this);
		}

		// ObjectInst
		ObjectInst objInst = getObjectInst(type, qualifiers);
		if (objInst == null) {
			throw new ObjectException("Object is required (type=" + type + ", qualifiers=" + Arrays.toString(qualifiers) + ")");
		}

		// Prototype?
		if (objInst.getScope() == ObjectScope.PROTOTYPE) {
			return ObjectUtils.cast(produceObject(objInst));
		}

		// Singleton
		if (objInst.getInstance() != null) {
			return ObjectUtils.cast(objInst.getInstance());
		}
		synchronized (this.mutex) {
			if (objInst.getInstance() == null) {
				objInst.setInstance(produceObject(objInst));
			}
			return ObjectUtils.cast(objInst.getInstance());
		}
	}

	public <T> T postConstruct(T obj) throws ObjectException {
		AssertUtils.assertNotNull(obj);
		invokeMethod(obj, PostConstruct.class);
		return obj;
	}

	public void preDestroy(Object obj) throws ObjectException {
		AssertUtils.assertNotNull(obj);
		invokeMethod(obj, PreDestroy.class);
	}

	@Override
	public void destroy() throws ObjectException {
		for (ObjectInst objInst : this.objectInsts) {
			if (objInst.getInstance() != null) {
				if (objInst.getProducer() == null) {
					preDestroy(objInst.getInstance());
				} else {
					objInst.getProducer().destroy(objInst.getInstance());
				}
				objInst.setInstance(null);
			}
		}
	}

	protected void invokeMethod(Object obj, Class<? extends Annotation> annotationType) throws ObjectException {
		Class<?> clazz = obj.getClass();
		while (clazz != Object.class) {
			Method[] methods = clazz.getDeclaredMethods();
			for (Method method : methods) {
				if (method.getDeclaredAnnotation(annotationType) != null) {
					try {
						method.setAccessible(true);
						method.invoke(obj);
						return;
					} catch (ObjectException ex) {
						throw ex;
					} catch (Exception ex) {
						throw new ObjectException(ex);
					}
				}
			}
			clazz = clazz.getSuperclass();
		}
	}

	public Iterator<ObjectDesc> getDescIterator() {
		return new Iterator<ObjectDesc>() {

			int index = -1;
			final Object[] objInsts = objectInsts.toArray();

			@Override
			public ObjectDesc next() {
				ObjectInst objInst = (ObjectInst) this.objInsts[++this.index];
				return new ObjectDesc(objInst.getType(), objInst.getQualifiers());
			}

			@Override
			public boolean hasNext() {
				return this.index < this.objInsts.length - 1;
			}

			@Override
			public void remove() {
				throw new UnsupportedOperationException();
			}
		};
	}

	protected static abstract class InjectTraverser {

		public abstract boolean isValidating();

		public abstract void onParameter(Parameter parameter) throws ObjectException;

		public abstract void onField(Field field) throws ObjectException;

		public abstract void onMethod(Method method) throws ObjectException;

		public void traverse(Class<?> implClass) throws ObjectException {
			Class<?> clazz = null;

			// Validating
			if (isValidating()) {

				// Constructor
				for (Constructor<?> ctor : implClass.getDeclaredConstructors()) {
					if (ctor.getDeclaredAnnotation(Inject.class) != null) {
						for (Parameter parameter : ctor.getParameters()) {
							onParameter(parameter);
						}
						break;
					}
				}

				// Method
				clazz = implClass;
				while (clazz != Object.class) {
					Method[] methods = clazz.getDeclaredMethods();
					for (Method method : methods) {
						if (method.getDeclaredAnnotation(Inject.class) != null) {
							for (Parameter parameter : method.getParameters()) {
								onParameter(parameter);
							}
						}
					}
					clazz = clazz.getSuperclass();
				}
			}

			// Fields
			clazz = implClass;
			while (clazz != Object.class) {
				Field[] fields = clazz.getDeclaredFields();
				for (Field field : fields) {
					if (field.getDeclaredAnnotation(Inject.class) != null) {
						onField(field);
					}
				}
				clazz = clazz.getSuperclass();
			}

			// Injecting?
			if (!isValidating()) {

				clazz = implClass;
				while (clazz != Object.class) {
					Method[] methods = clazz.getDeclaredMethods();
					for (Method method : methods) {
						if (method.getDeclaredAnnotation(Inject.class) != null) {
							onMethod(method);
						}
					}
					clazz = clazz.getSuperclass();
				}
			}
		}
	}

	protected static class KeyDesc {

		final Class<?> type;
		final Annotation[] qualifiers;

		public KeyDesc(Class<?> type, Annotation[] qualifiers) {
			this.type = type;
			this.qualifiers = qualifiers;
		}

		public Class<?> getType() {
			return this.type;
		}

		public Annotation[] getQualifiers() {
			return this.qualifiers;
		}

		@Override
		public int hashCode() {
			int hash = 1, p = 31;
			hash = p * hash + Objects.hashCode(this.type);
			hash = p * hash + Arrays.hashCode(this.qualifiers);
			return hash;
		}

		@Override
		public boolean equals(Object obj) {
			KeyDesc another = (KeyDesc) obj;
			return (this.type == another.type) && AnnotationUtils.equals(this.qualifiers, another.qualifiers);
		}
	}

	public static class ObjectDesc {

		final Class<?> type;
		final Annotation[] qualifiers;

		public ObjectDesc(Class<?> type, Annotation[] qualifiers) {
			this.type = type;
			this.qualifiers = qualifiers;
		}

		public Class<?> getType() {
			return this.type;
		}

		public Annotation[] getQualifiers() {
			return this.qualifiers;
		}

		@Override
		public int hashCode() {
			int hash = 1, p = 31;
			hash = p * hash + Objects.hashCode(this.type);
			hash = p * hash + Arrays.hashCode(this.qualifiers);
			return hash;
		}

		@Override
		public boolean equals(Object obj) {
			ObjectDesc another = (ObjectDesc) obj;
			return (this.type == another.type) && AnnotationUtils.equals(this.qualifiers, another.qualifiers);
		}

		@Override
		public String toString() {
			return "type=" + this.type + ", qualifiers=" + Arrays.toString(this.qualifiers);
		}
	}

	@SuppressWarnings("rawtypes")
	protected static class ObjectInst {

		private Class<?> type;
		private Annotation[] qualifiers;
		private ObjectScope scope;
		private ObjectProducer producer;
		private volatile Object instance;

		public Class<?> getType() {
			return this.type;
		}

		public ObjectInst setType(Class<?> type) {
			this.type = type;
			return this;
		}

		public Annotation[] getQualifiers() {
			return this.qualifiers;
		}

		public ObjectInst setQualifiers(Annotation[] qualifiers) {
			this.qualifiers = qualifiers;
			return this;
		}

		public ObjectScope getScope() {
			return scope;
		}

		public ObjectInst setScope(ObjectScope scope) {
			this.scope = scope;
			return this;
		}

		public ObjectProducer getProducer() {
			return this.producer;
		}

		public ObjectInst setProducer(ObjectProducer producer) {
			this.producer = producer;
			return this;
		}

		public Object getInstance() {
			return this.instance;
		}

		public ObjectInst setInstance(Object instance) {
			this.instance = instance;
			return this;
		}

		@Override
		public int hashCode() {
			int hash = 1, p = 31;
			hash = p * hash + Objects.hashCode(this.type);
			hash = p * hash + Arrays.hashCode(this.qualifiers);
			return hash;
		}

		@Override
		public boolean equals(Object obj) {
			ObjectInst another = (ObjectInst) obj;
			return (this.type == another.type) && AnnotationUtils.equals(this.qualifiers, another.qualifiers);
		}

		@Override
		public String toString() {
			return "type=" + this.type + ", qualifiers=" + Arrays.toString(this.qualifiers) + ", scope=" + this.scope + ", producer=" + this.producer;
		}
	}
}
