// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.formatters;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.utils.TypeUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class FormatterProvider extends InitializeObject {

	public static final String INSTANCE_ID = "__formatterProvider";

	private static final class FormatterProviderHolder {
		static final FormatterProvider DEFAULT = new FormatterProvider();
	}

	public static FormatterProvider getDefault() {
		return FormatterProviderHolder.DEFAULT;
	}

	private Map<String, Formatter> formatterMap = new HashMap<>();

	@Override
	protected void init() throws Exception {
		addDefault(Formatter.BYTE, new ByteFormatter());
		addDefault(Formatter.SHORT, new ShortFormatter());
		addDefault(Formatter.INTEGER, new IntegerFormatter());
		addDefault(Formatter.LONG, new LongFormatter());
		addDefault(Formatter.BIGINTEGER, new BigIntegerFormatter());

		addDefault(Formatter.FLOAT, new FloatFormatter());
		addDefault(Formatter.DOUBLE, new DoubleFormatter());
		addDefault(Formatter.BIGDECIMAL, new BigDecimalFormatter());

		addDefault(Formatter.SQL_DATE, new SqlDateFormatter());
		addDefault(Formatter.SQL_TIME, new SqlTimeFormatter());
		addDefault(Formatter.SQL_DATETIME, new SqlDateTimeFormatter());
		addDefault(Formatter.SQL_TIME_24H, new SqlTime24HFormatter());
		addDefault(Formatter.SQL_DATETIME_24H, new SqlDateTime24HFormatter());

		addDefault(Formatter.BOOLEAN, new BooleanFormatter());
		addDefault(Formatter.CHARACTER, new CharacterFormatter());

		this.formatterMap = Collections.unmodifiableMap(this.formatterMap);
	}

	protected void addDefault(String name, Formatter formatter) {
		if (!this.formatterMap.containsKey(name)) {
			this.formatterMap.put(name, formatter);
		}
	}

	public void addFormatter(String name, Formatter formatter) {
		this.assertNotInitialized();
		this.formatterMap.put(name, formatter);
	}

	public Formatter findFormatter(Class<?> targetType, FormatterNamed named) {
		this.initialize();
		if (named != null) {
			return getFormatter(named.value());
		} else {
			String name = TypeUtils.wrap(targetType).getSimpleName();
			return this.formatterMap.get(name);
		}
	}

	public Formatter getFormatter(Class<?> targetType) throws IllegalArgumentException {
		this.initialize();
		String name = TypeUtils.wrap(targetType).getSimpleName();
		Formatter formatter = this.formatterMap.get(name);
		if (formatter == null) {
			throw new IllegalArgumentException("Formatter is required (targetType=" + targetType + ")");
		}
		return formatter;
	}

	public Formatter getFormatter(String name) throws IllegalArgumentException {
		this.initialize();
		Formatter formatter = this.formatterMap.get(name);
		if (formatter == null) {
			throw new IllegalArgumentException("Formatter is required (name=" + name + ")");
		}
		return formatter;
	}
}
