// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.crypto;

import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.base.TokenGenerator;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.ValueUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class TokenDigester extends InitializeObject {

	private int tokenLength;
	private TokenGenerator generator;

	private TextDigester digester;

	public TokenDigester() {
	}

	public TokenDigester(TextDigester digester) {
		this.digester = digester;
	}

	public TokenDigester(TextDigester digester, int tokenLength) {
		this.digester = digester;
		this.tokenLength = tokenLength;
	}

	@Override
	protected void init() throws Exception {
		this.tokenLength = ValueUtils.valueOrDefault(this.tokenLength, 32);
		AssertUtils.assertTrue(this.tokenLength > 0, "tokenLength is required.");

		this.generator = new TokenGenerator(this.tokenLength);
		AssertUtils.assertNotNull(this.digester, "digester is required.");
	}

	public String generate() throws CryptoException {
		initialize();
		return this.generator.generate();
	}

	public String digest(String token) throws CryptoException {
		initialize();
		AssertUtils.assertNotNull(token, "token is required.");
		return this.digester.digest(token);
	}

	public boolean verify(String token, String digested) throws CryptoException {
		initialize();
		AssertUtils.assertNotNull(token, "token is required.");
		AssertUtils.assertNotNull(digested, "digested is required.");

		return this.digester.verify(token, digested);
	}

	public TokenDigester setTokenLength(int tokenLength) {
		this.assertNotInitialized();
		this.tokenLength = tokenLength;
		return this;
	}

	public TokenDigester setDigester(TextDigester digester) {
		this.assertNotInitialized();
		this.digester = digester;
		return this;
	}

	public TokenDigester copy() {
		TokenDigester impl = new TokenDigester().setTokenLength(this.tokenLength);
		if (this.digester != null) {
			impl.digester = this.digester.copy();
		}
		return impl;
	}
}
