// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.crypto;

import java.nio.charset.Charset;

import com.appslandia.common.base.DestroyException;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.BaseEncodingUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class TextDigester extends TextBasedCrypto {
	private Digester digester;

	public TextDigester() {
	}

	public TextDigester(Digester digester) {
		this.digester = digester;
	}

	@Override
	protected void init() throws Exception {
		super.init();
		AssertUtils.assertNotNull(this.digester, "digester is required.");
	}

	public String digest(String message) throws CryptoException {
		this.initialize();
		AssertUtils.assertNotNull(message, "message is required.");

		return BaseEncodingUtils.encode(this.digester.digest(message.getBytes(this.textCharset)), this.baseEncoding);
	}

	public boolean verify(String message, String digested) throws CryptoException {
		this.initialize();
		AssertUtils.assertNotNull(message, "message is required.");
		AssertUtils.assertNotNull(digested, "digested is required.");

		return this.digester.verify(message.getBytes(this.textCharset), BaseEncodingUtils.decode(digested, this.baseEncoding));
	}

	@Override
	public void destroy() throws DestroyException {
		if (this.digester != null) {
			this.digester.destroy();
		}
		super.destroy();
	}

	public TextDigester setDigester(Digester digester) {
		this.assertNotInitialized();
		this.digester = digester;
		return this;
	}

	@Override
	public TextDigester setTextCharset(Charset charset) {
		super.setTextCharset(charset);
		return this;
	}

	@Override
	public TextDigester setTextCharset(String textCharset) {
		super.setTextCharset(textCharset);
		return this;
	}

	@Override
	public TextDigester setBaseEncoding(String baseEncoding) {
		super.setBaseEncoding(baseEncoding);
		return this;
	}

	public TextDigester copy() {
		TextDigester impl = new TextDigester();
		impl.setTextCharset(this.textCharset).setBaseEncoding(this.baseEncoding);
		if (this.digester != null) {
			impl.digester = this.digester.copy();
		}
		return impl;
	}
}
