// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.crypto;

import java.nio.charset.Charset;

import com.appslandia.common.base.InitializeObject;
import com.appslandia.common.utils.AssertUtils;
import com.appslandia.common.utils.BaseEncodingUtils;
import com.appslandia.common.utils.CharsetUtils;
import com.appslandia.common.utils.StringUtils;
import com.appslandia.common.utils.ValueUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public abstract class TextBasedCrypto extends InitializeObject {

	protected Charset textCharset;
	protected String baseEncoding;

	@Override
	protected void init() throws Exception {
		this.textCharset = ValueUtils.valueOrDefault(this.textCharset, CharsetUtils.UTF_8);

		// baseEncoding
		this.baseEncoding = ValueUtils.valueOrDefault(this.baseEncoding, BaseEncodingUtils.ENCODING_BASE64);
		AssertUtils.assertTrue(BaseEncodingUtils.isEncodingSupported(this.baseEncoding), "baseEncoding is not supported.");
	}

	public TextBasedCrypto setTextCharset(Charset charset) {
		this.assertNotInitialized();
		this.textCharset = charset;
		return this;
	}

	public TextBasedCrypto setTextCharset(String textCharset) {
		this.assertNotInitialized();
		String cs = StringUtils.trimToNull(textCharset);
		if (cs != null) {
			this.textCharset = Charset.forName(cs);
		}
		return this;
	}

	public TextBasedCrypto setBaseEncoding(String baseEncoding) {
		this.assertNotInitialized();
		this.baseEncoding = StringUtils.trimToNull(baseEncoding);
		return this;
	}
}
