// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.crypto;

import java.text.Normalizer;

import javax.crypto.Cipher;

import com.appslandia.common.base.PropertyConfig;
import com.appslandia.common.utils.CharsetUtils;
import com.appslandia.common.utils.DateUtils;
import com.appslandia.common.utils.StringUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class SecureConfig extends PropertyConfig {
	private static final long serialVersionUID = 1L;

	final TextEncryptor textEncryptor;

	public SecureConfig() {
		char[] pw = CryptoUtils.toCharArray(Cipher.class.getName().getBytes(CharsetUtils.ISO_8859_1));
		char[] np = Normalizer.normalize(new String(pw), Normalizer.Form.NFC).toCharArray();
		try {
			this.textEncryptor = new TextEncryptor(new PbeEncryptor().setTransformation("AES/CBC/PKCS5Padding").setKeySize(16).setPassword(np));
		} finally {
			CryptoUtils.clear(np);
			CryptoUtils.clear(pw);
		}
	}

	public SecureConfig(char[] password) {
		this.textEncryptor = new TextEncryptor(new PbeEncryptor().setTransformation("AES/CBC/PKCS5Padding").setKeySize(16).setPassword(password));
	}

	public SecureConfig(Encryptor encryptor) {
		this.textEncryptor = new TextEncryptor(encryptor);
	}

	public SecureConfig(TextEncryptor textEncryptor) {
		this.textEncryptor = textEncryptor;
	}

	@Override
	public String get(Object key) throws CryptoException {
		String value = super.get(key);
		if (value == null) {
			return null;
		}
		if (!isEncryptedValue(value)) {
			return value;
		}
		return this.textEncryptor.decrypt(parseEncryptedValue(value));
	}

	protected void secure(String key, String value) throws CryptoException {
		this.configs.put(key, (value != null) ? markEncryptedValue(this.textEncryptor.encrypt(value)) : null);
	}

	public void sput(String key, String value) throws CryptoException {
		secure(key, StringUtils.trimToNull(value));
	}

	public void sput(String key, boolean value) throws CryptoException {
		secure(key, Boolean.toString(value));
	}

	public void sput(String key, int value) throws CryptoException {
		secure(key, Integer.toString(value));
	}

	public void sput(String key, long value) throws CryptoException {
		secure(key, Long.toString(value));
	}

	public void sput(String key, float value) throws CryptoException {
		secure(key, Float.toString(value));
	}

	public void sput(String key, double value) throws CryptoException {
		secure(key, Double.toString(value));
	}

	public void sput(String key, Boolean value) throws CryptoException {
		secure(key, (value != null) ? value.toString() : null);
	}

	public void sput(String key, Integer value) throws CryptoException {
		secure(key, (value != null) ? value.toString() : null);
	}

	public void sput(String key, Long value) throws CryptoException {
		secure(key, (value != null) ? value.toString() : null);
	}

	public void sput(String key, Float value) throws CryptoException {
		secure(key, (value != null) ? value.toString() : null);
	}

	public void sput(String key, Double value) throws CryptoException {
		secure(key, (value != null) ? value.toString() : null);
	}

	public void sput(String key, java.sql.Date value) throws CryptoException {
		secure(key, (value != null) ? DateUtils.iso8601Date(value) : null);
	}

	public void sput(String key, java.sql.Time value) throws CryptoException {
		secure(key, (value != null) ? DateUtils.iso8601Time(value) : null);
	}

	public void sput(String key, java.sql.Timestamp value) throws CryptoException {
		secure(key, (value != null) ? DateUtils.iso8601DateTime(value) : null);
	}

	private static final String ENC_BEGIN = "ENC(";
	private static final String ENC_END = ")";

	private static String markEncryptedValue(String value) {
		return ENC_BEGIN + value + ENC_END;
	}

	private static boolean isEncryptedValue(String value) {
		return value.startsWith(ENC_BEGIN) && value.endsWith(ENC_END);
	}

	private static String parseEncryptedValue(String value) {
		return value.substring(ENC_BEGIN.length(), value.length() - ENC_END.length());
	}
}
