// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.base;

import java.io.Reader;
import java.io.Writer;

import com.appslandia.common.utils.AssertUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public abstract class JsonProcessor extends InitializeObject {

	public abstract void write(Writer out, Object obj) throws JsonException;

	public abstract <T> T read(Reader reader, Class<? extends T> resultClass) throws JsonException;

	public abstract JsonProcessor copy();

	public String toString(Object obj) throws JsonException {
		StringWriter out = new StringWriter();
		write(out, obj);
		return out.toString();
	}

	private static JsonProcessor instance;
	private static final Object MUTEX = new Object();

	public static JsonProcessor getInstance() {
		JsonProcessor obj = instance;
		if (obj == null) {
			synchronized (MUTEX) {
				if ((obj = instance) == null) {
					instance = obj = initJsonProcessor();
				}
			}
		}
		AssertUtils.assertNotNull(obj, "JsonProcessor.instance is required.");
		return obj;
	}

	public static void setInstance(JsonProcessor instance) {
		AssertUtils.assertNull(JsonProcessor.instance);
		JsonProcessor.instance = instance;
	}

	private static JsonProcessor initJsonProcessor() {
		try {
			Class<?> implClass = null;
			final String gsonImpl = "com.appslandia.common.gson.GsonProcessor";
			try {
				ClassLoader cl = Thread.currentThread().getContextClassLoader();
				if (cl != null) {
					implClass = cl.loadClass(gsonImpl);
				}
			} catch (ClassNotFoundException ignore) {
			}
			if (implClass == null) {
				try {
					implClass = JsonProcessor.class.getClassLoader().loadClass(gsonImpl);
				} catch (ClassNotFoundException ignore) {
				}
			}
			if (implClass != null) {
				return JsonProcessor.class.cast(implClass.newInstance());
			}
			return null;
		} catch (Exception ex) {
			throw new InitializeException(ex);
		}
	}
}
