// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.base;

import java.io.Serializable;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.appslandia.common.utils.DateUtils;
import com.appslandia.common.utils.ObjectUtils;
import com.appslandia.common.utils.SplitUtils;
import com.appslandia.common.utils.StringFormatUtils;
import com.appslandia.common.utils.StringUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public abstract class ConfigBase implements Config, Map<String, String>, Serializable {
	private static final long serialVersionUID = 1L;

	protected final Map<String, String> configs;

	public ConfigBase(Map<String, String> newMap) {
		this.configs = newMap;
	}

	@Override
	public String getString(String key) {
		return get(key);
	}

	@Override
	public String getString(String key, String defaultValue) {
		String value = get(key);
		return (value != null) ? value : defaultValue;
	}

	@Override
	public String getRequiredString(String key) throws NullPointerException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		return value;
	}

	@Override
	public List<String> getStringList(String key) {
		String value = get(key);
		if (value == null) {
			return Collections.emptyList();
		}
		return Collections.unmodifiableList(SplitUtils.splitByComma(value));
	}

	@Override
	public String getFormatted(String key) {
		String value = get(key);
		if (value == null) {
			return null;
		}
		Map<String, Object> props = ObjectUtils.cast(this);
		return StringFormatUtils.format(value, props);
	}

	@Override
	public String getRequiredFormatted(String key) throws NullPointerException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		Map<String, Object> props = ObjectUtils.cast(this);
		return StringFormatUtils.format(value, props);
	}

	@Override
	public String getFormatted(String key, Map<String, Object> parameters) {
		String value = get(key);
		if (value == null) {
			return null;
		}
		return StringFormatUtils.format(value, parameters);
	}

	@Override
	public String getRequiredFormatted(String key, Map<String, Object> parameters) throws NullPointerException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		return StringFormatUtils.format(value, parameters);
	}

	@Override
	public String getFormatted(String key, Object... parameters) {
		String value = get(key);
		if (value == null) {
			return null;
		}
		return StringFormatUtils.format(value, parameters);
	}

	@Override
	public String getRequiredFormatted(String key, Object... parameters) throws NullPointerException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		return StringFormatUtils.format(value, parameters);
	}

	@Override
	public boolean getBool(String key, boolean defaultValue) {
		String value = get(key);
		if (value == null) {
			return defaultValue;
		}
		if (value.equalsIgnoreCase(Boolean.TRUE.toString())) {
			return true;
		}
		if (value.equalsIgnoreCase(Boolean.FALSE.toString())) {
			return false;
		}
		return defaultValue;
	}

	@Override
	public boolean getRequiredBool(String key) throws NullPointerException, BoolFormatException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		if (value.equalsIgnoreCase(Boolean.TRUE.toString())) {
			return true;
		}
		if (value.equalsIgnoreCase(Boolean.FALSE.toString())) {
			return false;
		}
		throw new BoolFormatException(value);
	}

	@Override
	public int getInt(String key, int defaultValue) {
		String value = get(key);
		if (value == null) {
			return defaultValue;
		}
		try {
			return Integer.parseInt(value);
		} catch (NumberFormatException ex) {
			return defaultValue;
		}
	}

	@Override
	public int getRequiredInt(String key) throws NullPointerException, NumberFormatException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		return Integer.parseInt(value);
	}

	@Override
	public long getLong(String key, long defaultValue) {
		String value = get(key);
		if (value == null) {
			return defaultValue;
		}
		try {
			return Long.parseLong(value);
		} catch (NumberFormatException ex) {
			return defaultValue;
		}
	}

	@Override
	public long getRequiredLong(String key) throws NullPointerException, NumberFormatException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		return Long.parseLong(value);
	}

	@Override
	public float getFloat(String key, float defaultValue) {
		String value = get(key);
		if (value == null) {
			return defaultValue;
		}
		try {
			return Float.parseFloat(value);
		} catch (NumberFormatException ex) {
			return defaultValue;
		}
	}

	@Override
	public float getRequiredFloat(String key) throws NullPointerException, NumberFormatException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		return Float.parseFloat(value);
	}

	@Override
	public double getDouble(String key, double defaultValue) {
		String value = get(key);
		if (value == null) {
			return defaultValue;
		}
		try {
			return Double.parseDouble(value);
		} catch (NumberFormatException ex) {
			return defaultValue;
		}
	}

	@Override
	public double getRequiredDouble(String key) throws NullPointerException, NumberFormatException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		return Double.parseDouble(value);
	}

	@Override
	public Date getDate(String key) {
		String value = get(key);
		if (value == null) {
			return null;
		}
		try {
			return DateUtils.iso8601Date(value);
		} catch (DateFormatException ex) {
			return null;
		}
	}

	@Override
	public java.sql.Date getRequiredDate(String key) throws NullPointerException, DateFormatException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		return DateUtils.iso8601Date(value);
	}

	@Override
	public Time getTime(String key) {
		String value = this.getString(key);
		if (value == null) {
			return null;
		}
		try {
			return DateUtils.iso8601Time(value);
		} catch (DateFormatException ex) {
			return null;
		}
	}

	@Override
	public Time getRequiredTime(String key) throws NullPointerException, DateFormatException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		return DateUtils.iso8601Time(value);
	}

	@Override
	public Timestamp getDateTime(String key) {
		String value = get(key);
		if (value == null) {
			return null;
		}
		try {
			return DateUtils.iso8601DateTime(value);
		} catch (DateFormatException ex) {
			return null;
		}
	}

	@Override
	public java.sql.Timestamp getRequiredDateTime(String key) throws NullPointerException, DateFormatException {
		String value = get(key);
		if (value == null) {
			throw new NullPointerException("Property is required (key=" + key + ")");
		}
		return DateUtils.iso8601DateTime(value);
	}

	@Override
	public String get(Object key) {
		return this.configs.get(key);
	}

	@Override
	public Set<String> keySet() {
		return this.configs.keySet();
	}

	@Override
	public int size() {
		return this.configs.size();
	}

	@Override
	public boolean isEmpty() {
		return this.configs.isEmpty();
	}

	@Override
	public boolean containsKey(Object key) {
		return this.configs.containsKey(key);
	}

	@Override
	public boolean containsValue(Object value) {
		throw new UnsupportedOperationException();
	}

	@Override
	public String put(String key, String value) {
		return this.configs.put(key, StringUtils.trimToNull(value));
	}

	@Override
	public String remove(Object key) {
		return this.configs.remove(key);
	}

	@Override
	public void putAll(Map<? extends String, ? extends String> m) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void clear() {
		this.configs.clear();
	}

	@Override
	public Collection<String> values() {
		throw new UnsupportedOperationException();
	}

	@Override
	public Set<Entry<String, String>> entrySet() {
		return this.configs.entrySet();
	}
}
