// The MIT License (MIT)
// Copyright © 2015 AppsLandia. All rights reserved.

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package com.appslandia.common.base;

import com.appslandia.common.utils.CharUtils;

/**
 *
 * @author <a href="mailto:haducloc13@gmail.com">Loc Ha</a>
 *
 */
public class Base16 {

	private static final char[] HEX_CHARS = CharUtils.toCharRanges("0-9A-F");

	private Base16() {
	}

	public static Encoder getEncoder() {
		return Encoder.INSTANCE;
	}

	public static Decoder getDecoder() {
		return Decoder.INSTANCE;
	}

	public static class Encoder {
		static final Encoder INSTANCE = new Encoder();

		private Encoder() {
		}

		public byte[] encode(byte[] src) {
			byte[] dest = new byte[src.length * 2];
			int i = 0;
			for (byte b : src) {
				dest[i++] = (byte) ((b & 0xf0) >> 4);
				dest[i++] = (byte) (b & 0x0f);
			}
			return dest;
		}

		public char[] encodeToChars(byte[] src) {
			char[] dest = new char[src.length * 2];
			int i = 0;
			for (byte b : src) {
				dest[i++] = HEX_CHARS[(b & 0xf0) >> 4];
				dest[i++] = HEX_CHARS[b & 0x0f];
			}
			return dest;
		}

		public String encodeToString(byte[] src) {
			return new String(encodeToChars(src));
		}
	}

	public static class Decoder {
		static final Decoder INSTANCE = new Decoder();

		private Decoder() {
		}

		public byte[] decode(byte[] src) {
			byte[] dest = new byte[src.length / 2];

			int i = 0;
			for (int j = 0; j < dest.length; j++) {
				byte v1 = src[j * 2];
				byte v2 = src[j * 2 + 1];

				if (v1 < 0 || v1 > 15) {
					throw new IllegalArgumentException("src is invalid base16.");
				}
				if (v2 < 0 || v2 > 15) {
					throw new IllegalArgumentException("src is invalid base16.");
				}
				dest[i++] = (byte) ((v1 << 4) + v2);
			}
			return dest;
		}

		public byte[] decodeFromChars(char[] src) {
			byte[] dest = new byte[src.length / 2];

			int i = 0;
			for (int j = 0; j < dest.length; j++) {
				int v1 = Character.digit(src[j * 2], 16);
				int v2 = Character.digit(src[j * 2 + 1], 16);

				if (v1 < 0 || v1 > 15) {
					throw new IllegalArgumentException("src is invalid base16.");
				}
				if (v2 < 0 || v2 > 15) {
					throw new IllegalArgumentException("src is invalid base16.");
				}
				dest[i++] = (byte) ((v1 << 4) + v2);
			}
			return dest;
		}

		public byte[] decodeFromString(String src) {
			return decodeFromChars(src.toCharArray());
		}
	}
}
