package com.applovin.sdk;


import android.content.Context;
import android.util.Log;

import com.applovin.impl.sdk.utils.Utils;

import java.util.HashMap;
import java.util.Map;

/**
 * This class contains settings for AppLovin SDK.
 *
 * @author Basil Shikin
 */
public class AppLovinSdkSettings
{
    private static final String TAG = "AppLovinSdkSettings";

    private boolean isTestAdsEnabled;
    private boolean isVerboseLoggingEnabled;
    private long    bannerAdRefreshSeconds;
    private String  autoPreloadSizes;
    private String  autoPreloadTypes;
    private boolean muted;

    private final Map<String, Object> localSettings = new HashMap<>(); // NOTE: do not rename `localSettings` - it is used internally via reflection.

    /**
     * @deprecated - Use {@link AppLovinSdkSettings#AppLovinSdkSettings(Context)} instead.
     */
    public AppLovinSdkSettings()
    {
        this( null );
    }

    /**
     * Creates an instance of AppLovin SDK's settings object with the given context to extract.
     */
    public AppLovinSdkSettings(final Context context)
    {
        this.isVerboseLoggingEnabled = Utils.isVerboseLoggingEnabled( context );
        this.isTestAdsEnabled = Utils.isTestAdsEnabled( context );
        this.bannerAdRefreshSeconds = -1;
        this.autoPreloadSizes = AppLovinAdSize.INTERSTITIAL.getLabel() + "," + AppLovinAdSize.BANNER.getLabel() + "," + AppLovinAdSize.MREC.getLabel();
        this.autoPreloadTypes = AppLovinAdType.INCENTIVIZED.getLabel() + "," + AppLovinAdType.REGULAR.getLabel() + "," + AppLovinAdType.NATIVE.getLabel();
    }

    /**
     * Toggle test ads for the SDK. If enabled, AppLovin will display test ads from our servers, guaranteeing 100% fill.
     * This is for integration testing only. Ensure that you set this to false when the app is launched.
     *
     * @param isTestAdsEnabled True if you want to receive test ads. False if you want live ads.
     */
    public void setTestAdsEnabled(boolean isTestAdsEnabled)
    {
        this.isTestAdsEnabled = isTestAdsEnabled;
    }

    /**
     * Check if test ads are enabled for the AppLovin SDK. If enabled AppLovin will display test ads from our servers, guaranteeing 100% fill.
     */
    public boolean isTestAdsEnabled()
    {
        return isTestAdsEnabled;
    }

    /**
     * Toggle verbose logging of AppLovin SDK. If enabled AppLovin messages will appear in standard application log accessible via logcat. All log messages will have "AppLovinSdk" tag.
     *
     * @param isVerboseLoggingEnabled True if log messages should be output.
     */
    public void setVerboseLogging(boolean isVerboseLoggingEnabled)
    {
        // If enabled from Android manifest, ignore programmatic setting.
        // This makes life easier for PubOps folks when mediation networks override this setting.
        if ( Utils.isVerboseLoggingConfigured() )
        {
            Log.e( TAG, "Ignoring setting of verbose logging - it is configured from Android manifest already or AppLovinSdkSettings was initialized without a context." );
        }
        else
        {
            this.isVerboseLoggingEnabled = isVerboseLoggingEnabled;
        }
    }

    /**
     * Check if verbose logging is enabled for the AppLovin SDK.
     * <p/>
     * <p/>
     * <p/>
     * If enabled AppLovin messages will appear in standard application log accessible via logcat. All log messages will have "AppLovinSdk" tag.
     */
    public boolean isVerboseLoggingEnabled()
    {
        return isVerboseLoggingEnabled;
    }

    /**
     * Whether video ads begin in a muted state or not. Defaults to `true`.
     */
    public boolean isMuted()
    {
        return muted;
    }

    /**
     * Set whether to begin video ads in a muted state or not. Defaults to `true`.
     *
     * @param muted If ads should begin in a muted state.
     */
    public void setMuted(boolean muted)
    {
        this.muted = muted;
    }

    /**
     * @deprecated Manually managing what ads SDK should automatically preload has been deprecated and will be removed in a future SDK version.
     */
    @Deprecated
    public void setAutoPreloadSizes(String autoPreloadSizes)
    {
        this.autoPreloadSizes = autoPreloadSizes;
    }

    @Deprecated
    public String getAutoPreloadSizes()
    {
        return autoPreloadSizes;
    }

    /**
     * @deprecated Manually managing what ads SDK should automatically preload has been deprecated and will be removed in a future SDK version.
     */
    @Deprecated
    public void setAutoPreloadTypes(String autoPreloadTypes)
    {
        this.autoPreloadTypes = autoPreloadTypes;
    }

    @Deprecated
    public String getAutoPreloadTypes()
    {
        return autoPreloadTypes;
    }

    /**
     * @deprecated Manually setting of banner ad refresh interval has been deprecated as automatic banner refresh has been disabled. This method will be removed in a future SDK version.
     */
    @Deprecated
    public long getBannerAdRefreshSeconds()
    {
        return bannerAdRefreshSeconds;
    }

    /**
     * @deprecated Manually setting of banner ad refresh interval has been deprecated as automatic banner refresh has been disabled. This method will be removed in a future SDK version.
     */
    @Deprecated
    public void setBannerAdRefreshSeconds(long adRefreshSeconds)
    {
        this.bannerAdRefreshSeconds = adRefreshSeconds;
    }

    @Override
    public String toString()
    {
        return "AppLovinSdkSettings{" +
                "isTestAdsEnabled=" + isTestAdsEnabled +
                ", isVerboseLoggingEnabled=" + isVerboseLoggingEnabled +
                ", muted=" + muted +
                '}';
    }
}
