package com.applovin.sdk;

import android.content.Context;
import android.util.Log;

import com.applovin.impl.sdk.CoreSdk;
import com.applovin.impl.sdk.Logger;
import com.applovin.impl.sdk.utils.Utils;
import com.applovin.nativeAds.AppLovinNativeAdService;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This is a base class for AppLovin Ad SDK.
 *
 * @author Basil Shikin
 */
public final class AppLovinSdk
{
    /**
     * Current SDK version.
     */
    public static final String VERSION      = BuildConfig.VERSION_NAME;
    public static final int    VERSION_CODE = BuildConfig.VERSION_CODE;

    private static final String TAG = "AppLovinSdk";

    private static       AppLovinSdk[] sdkInstances     = new AppLovinSdk[0];
    private static final Object        sdkInstancesLock = new Object();

    private final CoreSdk mSdkImpl; // NOTE: Do not rename `mSdkImpl` - we retrieve this internally via reflection

    /**
     * Listener interface to be used with {@link AppLovinSdk#initializeSdk(SdkInitializationListener)}
     */
    public interface SdkInitializationListener
    {
        void onSdkInitialized(AppLovinSdkConfiguration config);
    }

    /**
     * Get client SDK key.
     *
     * @return Client SDK key.
     * @hide
     */
    public String getSdkKey()
    {
        return mSdkImpl.getSdkKey();
    }

    /**
     * Get SDK settings provided on initialization
     *
     * @return Provided settings.
     */
    public AppLovinSdkSettings getSettings()
    {
        return mSdkImpl.getSettings();
    }

    /**
     * Set Plugin version.
     *
     * @param version Plugin version to set.
     */
    public void setPluginVersion(String version)
    {
        mSdkImpl.setPluginVersion( version );
    }

    /**
     * Set mediation provider using one of the provided strings above specified by {@link AppLovinMediationProvider}, or your own if not defined.
     *
     * @param mediationProvider The name of the mediation provider.
     */
    public void setMediationProvider(String mediationProvider)
    {
        mSdkImpl.setMediationProvider( mediationProvider );
    }

    /**
     * Get the mediation provider that was last set using {@link AppLovinSdk#setMediationProvider(String)}, or null if none was set.
     *
     * @return The mediation provider that was last set, or null if none was set.
     */
    public String getMediationProvider()
    {
        return mSdkImpl.getMediationProvider();
    }

    /**
     * Get an instance of the AppLovin Ad service. This service is used to fetch ads from AppLovin servers.
     *
     * @return Ad service. Guaranteed not to be null.
     */
    public AppLovinAdService getAdService()
    {
        return mSdkImpl.getAdService();
    }

    /**
     * Get an instance of the AppLovin native ad service. This service is used to fetch in-content ads from AppLovin servers.
     *
     * @return Ad service. Guaranteed not to be null.
     */
    public AppLovinNativeAdService getNativeAdService()
    {
        return mSdkImpl.getNativeAdService();
    }

    /**
     * Get an instance of the AppLovin postback service. This service is used to dispatch HTTP GET postbacks to arbitrary URLs.
     *
     * @return Postback service. Guaranteed not to be null.
     */
    public AppLovinPostbackService getPostbackService()
    {
        return mSdkImpl.getPostbackService();
    }

    /**
     * Get an instance of the AppLovin event service. This service is used to track post-install user events.
     *
     * @return Event service. Guaranteed not to be null.
     */
    public AppLovinEventService getEventService()
    {
        return mSdkImpl.getEventService();
    }

    /**
     * Get an instance of the AppLovin user service object for performing user-related tasks.
     *
     * @return User service. Guaranteed not to be null.
     */
    public AppLovinUserService getUserService()
    {
        return mSdkImpl.getUserService();
    }

    /**
     * Get an instance of the AppLovin variable service. This service is used to perform various AB tests that you have set up on your AppLovin dashboard on your users.
     *
     * @return Variable service. Guaranteed not to be null.
     */
    public AppLovinVariableService getVariableService()
    {
        return mSdkImpl.getVariableService();
    }

    /**
     * Set an identifier for the current user. This identifier will be tied to SDK events and our optional S2S postbacks.
     * <p>
     * If you're using reward validation, you can optionally set an identifier to be included with currency validation postbacks.
     * For example, a username or email. We'll include this in the postback when we ping your currency endpoint from our server.
     *
     * @param userIdentifier The user identifier to be set.
     */
    public void setUserIdentifier(String userIdentifier)
    {
        mSdkImpl.setUserIdentifier( userIdentifier );
    }

    /**
     * An identifier for the current user. This identifier will be tied to SDK events and our optional S2S postbacks.
     * <p>
     * If you're using reward validation, you can optionally set an identifier to be included with currency validation postbacks.
     * For example, a username or email. We'll include this in the postback when we ping your currency endpoint from our server.
     */
    public String getUserIdentifier()
    {
        return mSdkImpl.getUserIdentifier();
    }

    /**
     * Present the mediation debugger UI.
     * This debugger tool provides the status of your integration for each third-party ad network.
     *
     * Please call this method after the SDK has initialized, e.g. {@link AppLovinSdk#initializeSdk(SdkInitializationListener)}.
     */
    public void showMediationDebugger()
    {
        mSdkImpl.showMediationDebugger();
    }

    /**
     * Initialize the SDK
     */
    public void initializeSdk()
    {
        // Syntactic sugar
    }

    /**
     * Initialize the SDK with a given listener.
     * <p>
     * The callback will be invoked on the main thread.
     *
     * @param listener The callback that will be run when the SDK finishes initializing. May be null.
     */
    public void initializeSdk(SdkInitializationListener listener)
    {
        mSdkImpl.initializeSdk( listener );
    }

    /**
     * Initialize the default version of the SDK.
     * <p>
     * Please make sure that <code>AndroidManifest.xml</code> includes following line:
     * <p>
     * <pre>
     *     &lt;application>
     *                     . . .
     *         &lt;meta-data android:name="applovin.sdk.key" android:value="APPLOVIN_SDK_KEY" />
     *     &lt;/application>
     * </pre>
     *
     * @param context Android application context. Must not be null.
     */
    public static void initializeSdk(Context context)
    {
        initializeSdk( context, null );
    }

    /**
     * Initialize the default version of the SDK.
     * <p>
     * Please make sure that <code>AndroidManifest.xml</code> includes following line:
     * <p>
     * <pre>
     *     &lt;application>
     *                     . . .
     *         &lt;meta-data android:name="applovin.sdk.key" android:value="APPLOVIN_SDK_KEY" />
     *     &lt;/application>
     * </pre>
     *
     * @param context  Android application context. Must not be null.
     * @param listener The callback that will be run when the SDK finishes initializing. May be null.
     */
    public static void initializeSdk(Context context, SdkInitializationListener listener)
    {
        // Check input
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        final AppLovinSdk sdk = getInstance( context );
        if ( sdk != null )
        {
            sdk.initializeSdk( listener );
        }
        else
        {
            Log.e( TAG, "Unable to initialize AppLovin SDK: SDK object not created" );
        }
    }

    /**
     * Get the SDK configuration object provided upon initialization.
     *
     * @return An instance of the SDK configuration.
     */
    public AppLovinSdkConfiguration getConfiguration()
    {
        return mSdkImpl.getConfiguration();
    }

    /**
     * Get instance of AppLovin SDK that is configured in <code>AndroidManifest.xml</code>. Please make sure that <code>AndroidManifest.xml</code> includes following line:
     * <p>
     * <pre>
     *     &lt;application>
     *                     . . .
     *         &lt;meta-data android:value="YOUR_SDK_KEY_HERE" android:name="APPLOVIN_SDK_KEY" />
     *     &lt;/application>
     * </pre>
     *
     * @param context Android application context. Must not be null.
     *
     * @return An instance of AppLovinSDK
     */
    public static AppLovinSdk getInstance(final Context context)
    {
        return getInstance( new AppLovinInternalSdkSettings( context ), context );
    }

    /**
     * Get instance of AppLovin SDK that is configured in <code>AndroidManifest.xml</code>. Please make sure that <code>AndroidManifest.xml</code> includes following line:
     * <p>
     * <pre>
     *     &lt;application>
     *                     . . .
     *         &lt;meta-data android:value="YOUR_SDK_KEY_HERE" android:name="applovin.sdk.key" />
     *     &lt;/application>
     * </pre>
     *
     * @param settings Settings to use with an SDK. Must not be null.
     * @param context  Android application context. Must not be null.
     *
     * @return An instance of AppLovinSDK
     */
    public static AppLovinSdk getInstance(final AppLovinSdkSettings settings, final Context context)
    {
        // Check input
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        final String sdkKey = Utils.retrieveSdkKey( context );
        return getInstance( sdkKey, settings, context );
    }

    /**
     * Get an instance of AppLovin SDK.
     *
     * @param sdkKey       Client SDK's key.
     * @param userSettings User-provided settings. May be null.
     * @param context      Android context. Must not be null. Note that a reference to <code>context.getApplicationContext()</code> will be persisted by the API.
     *
     * @return An instance of AppLovinSDK
     */
    public static AppLovinSdk getInstance(final String sdkKey, final AppLovinSdkSettings userSettings, final Context context)
    {
        // Check input
        if ( userSettings == null ) throw new IllegalArgumentException( "No userSettings specified" );
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        synchronized ( sdkInstancesLock )
        {
            if ( sdkInstances.length == 1 && sdkInstances[0].getSdkKey().equals( sdkKey ) )
            {
                return sdkInstances[0];
            }
            else
            {
                for ( AppLovinSdk possibleMatch : sdkInstances )
                {
                    if ( possibleMatch.getSdkKey().equals( sdkKey ) )
                    {
                        return possibleMatch;
                    }
                }

                AppLovinSdk newSdk;
                try
                {
                    final CoreSdk newImpl = new CoreSdk();
                    newImpl.initialize( sdkKey, userSettings, context );
                    newSdk = new AppLovinSdk( newImpl );
                    newImpl.setWrappingSdk( newSdk );
                }
                catch ( Throwable th )
                {
                    Log.e( TAG, "Failed to build AppLovin SDK. Try cleaning application data and starting the application again.", th );
                    throw new RuntimeException( "Unable to build AppLovin SDK" );
                }

                final AppLovinSdk[] newSdkInstances = new AppLovinSdk[sdkInstances.length + 1];
                System.arraycopy( sdkInstances, 0, newSdkInstances, 0, sdkInstances.length );
                newSdkInstances[sdkInstances.length] = newSdk;
                sdkInstances = newSdkInstances;

                return newSdk;
            }
        }
    }

    /**
     * Check if SDK is enabled. SDK is enabled when (1) client successfully registered with the AppLovin server and (2) client is not banned from the server.
     *
     * @return True if SDK is ready to be used.
     */
    public boolean isEnabled()
    {
        return mSdkImpl.isEnabled();
    }

    /**
     * Check if SDK has a critical error prior to initialization.
     *
     * @return True if API is disabled because of a critical error.
     */
    public boolean hasCriticalErrors()
    {
        return mSdkImpl.hasCriticalErrors();
    }

    /**
     * Get context of a current application. This is a context the SDK was initialized with.
     * <p>
     * <p>
     * <b>Please note:</b> This method is for internal use only.
     * </p>
     *
     * @hide
     */
    Context getApplicationContext()
    {
        return mSdkImpl.getApplicationContext();
    }

    /**
     * Get an instance of SDK logger.
     * <p>
     * <p>
     * <b>Please note:</b> This method is for internal use only.
     * </p>
     *
     * @return SDK logger.
     * @hide
     */
    public Logger getLogger()
    {
        return mSdkImpl.getLogger();
    }

    private AppLovinSdk(CoreSdk sdk)
    {
        // Check input
        if ( sdk == null ) throw new IllegalArgumentException( "No sdk specified" );

        this.mSdkImpl = sdk;
    }

    /**
     * Re-initializes all SDK instances.
     * <p>
     * <b>Please note:</b> This method is for internal use only.
     * </p>
     *
     * @hide
     */
    static void reinitializeAll()
    {
        reinitializeAll( null );
    }

    /**
     * Re-initializes all SDK instances for when user consent has been explicitly granted.
     * <p>
     * <b>Please note:</b> This method is for internal use only.
     * </p>
     *
     * @hide
     */
    static void reinitializeAll(final Boolean hasUserConsent)
    {
        synchronized ( sdkInstancesLock )
        {
            for ( AppLovinSdk sdk : sdkInstances )
            {
                sdk.mSdkImpl.reinitialize();

                // Fire an event if user consent has been explicitly granted
                if ( hasUserConsent != null && hasUserConsent )
                {
                    Map<String, String> parameters = new HashMap<>( 1 );
                    parameters.put( "value", hasUserConsent.toString() );

                    sdk.getEventService().trackEvent( "huc", parameters );
                }
            }
        }
    }

    /**
     * All the AppLovin SDK instances currently in memory.
     *
     * @hide
     */
    public static List<AppLovinSdk> a()
    {
        return Arrays.asList( sdkInstances );
    }

    /**
     * Internal SDK settings class.
     * <p>
     * <b>Please note:</b> This class is for internal use only.
     * </p>
     *
     * @hide
     */
    private static class AppLovinInternalSdkSettings
            extends AppLovinSdkSettings
    {
        public AppLovinInternalSdkSettings(final Context context)
        {
            super( context );
        }
    }

    @Override
    public String toString()
    {
        return "AppLovinSdk{" +
                "sdkKey='" + getSdkKey() + "\'" +
                ", isEnabled=" + isEnabled() +
                ", isFirstSession=" + mSdkImpl.isFirstSession() +
                '}';
    }
}
