package com.applovin.sdk;

import java.util.HashSet;
import java.util.Locale;
import java.util.Set;


/**
 * This class defines a categorization of the behavior of an ad.
 */

public class AppLovinAdType
{
    /**
     * Represents a standard advertisement that does not provide a reward to the user.
     */
    public static final AppLovinAdType REGULAR = new AppLovinAdType( "REGULAR" );

    /**
     * Represents a rewarded video which will provide the user virtual currency upon completion.
     */
    public static final AppLovinAdType INCENTIVIZED = new AppLovinAdType( "VIDEOA" );

    /**
     * Represents a native ad.
     */
    public static final AppLovinAdType NATIVE = new AppLovinAdType( "NATIVE" );

    private final String label;

    /**
     * @hide
     **/
    public AppLovinAdType(String label)
    {
        this.label = label;
    }

    /**
     * Generate a String which contains the name of this ad type.
     *
     * @return Either "REGULAR" or "INCENTIVIZED".
     **/
    public String getLabel()
    {
        return label.toUpperCase( Locale.ENGLISH );
    }

    /**
     * @hide
     **/
    public static AppLovinAdType fromString(String type)
    {
        String typeStr = type.toUpperCase( Locale.ENGLISH );

        if ( typeStr.equals( INCENTIVIZED.getLabel() ) )
        {
            return INCENTIVIZED;
        }

        return REGULAR;
    }

    /**
     * Obtain a set of all available Ad Types.
     *
     * @return A set containing all available Ad Type objects.
     **/
    public static Set<AppLovinAdType> allTypes()
    {
        final Set<AppLovinAdType> allTypes = new HashSet<>( 2 );

        allTypes.add( AppLovinAdType.REGULAR );
        allTypes.add( AppLovinAdType.INCENTIVIZED );

        return allTypes;
    }

    /**
     * Simply calls <code>getLabel()</code>.
     *
     * @return Name of the ad type.
     */
    @Override
    public String toString()
    {
        return getLabel();
    }

    /**
     * @hide
     **/
    @Override
    public boolean equals(Object o)
    {
        // Generated by Android Studio
        if ( this == o ) return true;
        if ( o == null || getClass() != o.getClass() ) return false;

        AppLovinAdType that = (AppLovinAdType) o;

        return !( label != null ? !label.equals( that.label ) : that.label != null );

    }

    /**
     * @hide
     **/
    @Override
    public int hashCode()
    {
        // Generated by Android Studio
        return label != null ? label.hashCode() : 0;
    }
}
