package com.applovin.mediation.ads;

import android.app.Activity;
import android.text.TextUtils;

import com.applovin.impl.mediation.ads.MaxInterstitialImpl;
import com.applovin.mediation.MaxAdListener;
import com.applovin.mediation.MaxRewardedAdListener;
import com.applovin.sdk.AppLovinSdk;

/**
 * This class represents a full screen interstitial ad.
 */
public class MaxInterstitialAd
{
    private final MaxInterstitialImpl mImplementation;

    /**
     * Create a new Max interstitial.
     *
     * @param adUnitId Ad unit id to load ads for. Must not be null.
     * @param activity Current activity. Must not be null.
     */
    public MaxInterstitialAd(final String adUnitId, final Activity activity)
    {
        this( adUnitId, AppLovinSdk.getInstance( activity ), activity );
    }

    /**
     * Create a new Max interstitial.
     *
     * @param adUnitId Ad unit id to load ads for. Must not be null.
     * @param sdk      SDK to use. Must not be null. An instance of the SDK may be obtained by calling <code>AppLovinSdk.getInstance()</code>.
     * @param activity Current activity. Must not be null.
     */
    public MaxInterstitialAd(final String adUnitId, final AppLovinSdk sdk, final Activity activity)
    {
        // Check input
        if ( adUnitId == null ) throw new IllegalArgumentException( "No ad unit ID specified" );
        if ( TextUtils.isEmpty( adUnitId ) ) throw new IllegalArgumentException( "Empty ad unit ID specified" );
        if ( activity == null ) throw new IllegalArgumentException( "No activity specified" );
        if ( sdk == null ) throw new IllegalArgumentException( "No sdk specified" );

        mImplementation = new MaxInterstitialImpl( adUnitId, sdk, activity );
    }

    /**
     * Set a listener that will be notified about ad events.
     *
     * @param listener Listener to be notified. May be null.
     */
    public void setListener(final MaxAdListener listener)
    {
        mImplementation.setListener( listener );
    }

    /**
     * Set an extra parameter to pass to the server.
     *
     * @param key   Parameter key. Must not be null.
     * @param value Parameter value. May be null.
     */
    public void setExtraParameter(final String key, final String value)
    {
        mImplementation.setExtraParameter( key, value );
    }

    /**
     * Load ad for the current interstitial. Use {@link MaxInterstitialAd#setListener(MaxAdListener)} to assign a listener that should be
     * notified about ad load state.
     */
    public void loadAd()
    {
        mImplementation.loadAd();
    }

    /**
     * Show the loaded interstitial.
     * <p>
     * Use {@link MaxInterstitialAd#setListener(MaxAdListener)} to assign a listener that should be notified about display events.
     * Use {@link MaxInterstitialAd#isReady()} to check if an ad was successfully loaded.
     */
    public void showAd()
    {
        showAd( null );
    }

    /**
     * Show the loaded interstitial ad for a given placement to tie ad events to.
     * <p>
     * Use {@link MaxRewardedAd#setListener(MaxRewardedAdListener)} to assign a listener that should be notified about display events.
     * Use {@link MaxRewardedAd#isReady()} to check if an ad was successfully loaded.
     *
     * @param placement The placement to tie the showing ad's events to.
     */
    public void showAd(final String placement)
    {
        mImplementation.showAd( placement );
    }

    /**
     * Check if this ad is ready to be shown.
     *
     * @return <code>true</code> if the ad is ready to be shown.
     */
    public boolean isReady()
    {
        return mImplementation.isReady();
    }

    /**
     * Destroy current ad and fully remove it from memory.
     */
    public void destroy()
    {
        mImplementation.destroy();
    }

    @Override
    public String toString()
    {
        return mImplementation.toString();
    }
}
