package com.applovin.sdk;

import android.content.Context;

import com.applovin.impl.sdk.Logger;
import com.applovin.impl.sdk.utils.StringUtils;
import com.applovin.impl.sdk.utils.Utils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class contains settings for AppLovin SDK.
 *
 * @author Basil Shikin
 */
public class AppLovinSdkSettings
{
    private static final String TAG = "AppLovinSdkSettings";

    private boolean isVerboseLoggingEnabled;
    private long    bannerAdRefreshSeconds;
    private boolean muted;
    private boolean adInfoButtonEnabled;
    private boolean exceptionHandlerEnabled;

    private final Map<String, Object> localSettings            = new HashMap<>(); // NOTE: do not rename `localSettings` - it is used internally via reflection.
    private final Map<String, String> metaData                 = new HashMap<>(); // NOTE: on not rename `metaData` - it is used internally via reflection.
    private       List<String>        testDeviceAdvertisingIds = Collections.emptyList();
    private       List<String>        initializationAdUnitIds  = Collections.emptyList();

    /**
     * @deprecated - Use {@link AppLovinSdkSettings#AppLovinSdkSettings(Context)} instead.
     */
    public AppLovinSdkSettings()
    {
        this( null );
    }

    /**
     * Creates an instance of AppLovin SDK's settings object with the given context to extract.
     */
    public AppLovinSdkSettings(final Context context)
    {
        this.isVerboseLoggingEnabled = Utils.isVerboseLoggingEnabled( context );
        this.bannerAdRefreshSeconds = -1;
        this.exceptionHandlerEnabled = true;
    }

    /**
     * Enable devices to receive test ads, by passing in the advertising identifier (GAID) of each test device.
     * Refer to AppLovin logs for the GAID of your current device.
     */
    public void setTestDeviceAdvertisingIds(final List<String> testDeviceAdvertisingIds)
    {
        // Sanitize input and make copy of the list
        if ( testDeviceAdvertisingIds != null )
        {
            final List<String> sanitized = new ArrayList<>( testDeviceAdvertisingIds.size() );

            for ( final String advertisingId : testDeviceAdvertisingIds )
            {
                if ( advertisingId != null && advertisingId.length() == 36 )
                {
                    sanitized.add( advertisingId );
                }
                else
                {
                    Logger.userError( TAG, "Unable to set test device advertising id (" + advertisingId + ") - please make sure it is in the format of xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx" );
                }
            }

            this.testDeviceAdvertisingIds = sanitized;
        }
        else
        {
            this.testDeviceAdvertisingIds = Collections.emptyList();
        }
    }

    /**
     * Get the list of advertising identifiers that will receive test ads.
     */
    public List<String> getTestDeviceAdvertisingIds()
    {
        return testDeviceAdvertisingIds;
    }

    /**
     * Set the MAX ad unit ids that will be used for this instance of the SDK. 3rd-party SDKs will be initialized with the credentials configured for these ad unit ids.
     */
    public void setInitializationAdUnitIds(final List<String> initializationAdUnitIds)
    {
        // Sanitize input and make copy of the list
        if ( initializationAdUnitIds != null )
        {
            final List<String> sanitized = new ArrayList<>( initializationAdUnitIds.size() );

            for ( final String initializationAdUnitId : initializationAdUnitIds )
            {
                // Check for empty string from upstream deserialization of "" into [""], ignore...
                if ( StringUtils.isValidString( initializationAdUnitId ) && initializationAdUnitId.length() > 0 )
                {
                    // Correct length
                    if ( initializationAdUnitId.length() == 16 )
                    {
                        sanitized.add( initializationAdUnitId );
                    }
                    // Incorrect length
                    else
                    {
                        Logger.userError( TAG, "Unable to set initialization ad unit id (" + initializationAdUnitId + ") - please make sure it is in the format of XXXXXXXXXXXXXXXX" );
                    }
                }
            }

            this.initializationAdUnitIds = sanitized;
        }
        else
        {
            this.initializationAdUnitIds = Collections.emptyList();
        }
    }

    /**
     * Get the list of MAX ad unit ids that will be used for this instance of the SDK.
     */
    public List<String> getInitializationAdUnitIds()
    {
        return initializationAdUnitIds;
    }

    /**
     * Toggle verbose logging of AppLovin SDK. If enabled AppLovin messages will appear in standard application log accessible via logcat. All log messages will have "AppLovinSdk" tag.
     *
     * @param isVerboseLoggingEnabled True if log messages should be output.
     */
    public void setVerboseLogging(boolean isVerboseLoggingEnabled)
    {
        // If enabled from Android manifest, ignore programmatic setting.
        // This makes life easier for PubOps folks when mediation networks override this setting.
        if ( Utils.isVerboseLoggingConfigured() )
        {
            Logger.userError( TAG, "Ignoring setting of verbose logging - it is configured from Android manifest already or AppLovinSdkSettings was initialized without a context." );
            if ( Utils.isVerboseLoggingEnabled( null ) != isVerboseLoggingEnabled )
            {
                Logger.userError( TAG, "Attempted to programmatically set verbose logging flag to value different from value configured in Android Manifest." );
            }
        }
        else
        {
            this.isVerboseLoggingEnabled = isVerboseLoggingEnabled;
        }
    }

    /**
     * Check if verbose logging is enabled for the AppLovin SDK.
     * <p/>
     * If enabled AppLovin messages will appear in standard application log accessible via logcat. All log messages will have "AppLovinSdk" tag.
     */
    public boolean isVerboseLoggingEnabled()
    {
        return isVerboseLoggingEnabled;
    }

    /**
     * Whether video ads begin in a muted state or not. Defaults to {@code false}.
     */
    public boolean isMuted()
    {
        return muted;
    }

    /**
     * Set whether to begin video ads in a muted state or not.
     *
     * @param muted If ads should begin in a muted state.
     */
    public void setMuted(boolean muted)
    {
        this.muted = muted;
    }

    /**
     * Set whether the ad info button should be displayed on fullscreen ads (debug builds only).
     */
    public void setAdInfoButtonEnabled(boolean adInfoButtonEnabled)
    {
        this.adInfoButtonEnabled = adInfoButtonEnabled;
    }

    /**
     * Whether the ad info button will be displayed on fullscreen ads (debug builds only).
     */
    public boolean isAdInfoButtonEnabled()
    {
        return adInfoButtonEnabled;
    }

    /**
     * Set whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
     */
    public void setExceptionHandlerEnabled(boolean exceptionHandlerEnabled)
    {
        this.exceptionHandlerEnabled = true;
    }

    /**
     * Whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
     */
    public boolean isExceptionHandlerEnabled()
    {
        return exceptionHandlerEnabled;
    }

    /**
     * @deprecated Manually setting of banner ad refresh interval has been deprecated as automatic banner refresh has been disabled. This method will be removed in a future SDK version.
     */
    @Deprecated
    public long getBannerAdRefreshSeconds()
    {
        return bannerAdRefreshSeconds;
    }

    /**
     * @deprecated Manually setting of banner ad refresh interval has been deprecated as automatic banner refresh has been disabled. This method will be removed in a future SDK version.
     */
    @Deprecated
    public void setBannerAdRefreshSeconds(long adRefreshSeconds)
    {
        this.bannerAdRefreshSeconds = adRefreshSeconds;
    }

    @Override
    public String toString()
    {
        return "AppLovinSdkSettings{" +
                "isVerboseLoggingEnabled=" + isVerboseLoggingEnabled +
                ", muted=" + muted +
                ", testDeviceAdvertisingIds=" + testDeviceAdvertisingIds.toString() +
                ", initializationAdUnitIds=" + initializationAdUnitIds.toString() +
                ", adInfoButtonEnabled=" + adInfoButtonEnabled +
                ", exceptionHandlerEnabled=" + exceptionHandlerEnabled +
                '}';
    }
}
