package com.applovin.sdk;

import java.util.HashSet;
import java.util.Locale;
import java.util.Set;


/**
 * This class defines a categorization of the behavior of an ad.
 */
public class AppLovinAdType
{
    /**
     * Represents a standard advertisement that does not provide a reward to the user.
     */
    public static final AppLovinAdType REGULAR = new AppLovinAdType( "REGULAR" );

    /**
     * Represents a rewarded ad which will provide the user virtual currency upon completion.
     */
    public static final AppLovinAdType INCENTIVIZED = new AppLovinAdType( "VIDEOA" );

    /**
     * Represents a rewarded interstitial ad which the user can skip and be granted a reward upon successful completion of the ad.
     */
    public static final AppLovinAdType AUTO_INCENTIVIZED = new AppLovinAdType( "AUTOREW" );

    /**
     * Represents a native ad.
     */
    public static final AppLovinAdType NATIVE = new AppLovinAdType( "NATIVE" );

    private final String label;

    /**
     * @hide
     **/
    private AppLovinAdType(String label)
    {
        this.label = label;
    }

    /**
     * Generate a String which contains the name of this ad type.
     *
     * @return Either "REGULAR" or "INCENTIVIZED".
     **/
    public String getLabel()
    {
        return label.toUpperCase( Locale.ENGLISH );
    }

    /**
     * @hide
     **/
    public static AppLovinAdType fromString(String type)
    {
        if ( "REGULAR".equalsIgnoreCase( type ) )
        {
            return REGULAR;
        }
        else if ( "VIDEOA".equalsIgnoreCase( type ) )
        {
            return INCENTIVIZED;
        }
        else if ( "AUTOREW".equalsIgnoreCase( type ) )
        {
            return AUTO_INCENTIVIZED;
        }
        else if ( "NATIVE".equalsIgnoreCase( type ) )
        {
            return NATIVE;
        }
        else
        {
            throw new IllegalArgumentException( "Unknown Ad Type: " + type );
        }
    }

    /**
     * Obtain a set of all available Ad Types.
     *
     * @return A set containing all available Ad Type objects.
     **/
    public static Set<AppLovinAdType> allTypes()
    {
        final Set<AppLovinAdType> allTypes = new HashSet<>( 3 );

        allTypes.add( AppLovinAdType.REGULAR );
        allTypes.add( AppLovinAdType.INCENTIVIZED );
        allTypes.add( AppLovinAdType.AUTO_INCENTIVIZED );

        return allTypes;
    }

    /**
     * Simply calls <code>getLabel()</code>.
     *
     * @return Name of the ad type.
     */
    @Override
    public String toString()
    {
        return getLabel();
    }
}
