package com.applovin.mediation;

import android.app.Activity;
import android.util.DisplayMetrics;
import android.view.Display;

import com.applovin.impl.mediation.utils.MediationUtils;
import com.applovin.sdk.AppLovinSdkUtils;

/**
 * This class defines a format of an ad.
 */
public class MaxAdFormat
{
    /**
     * Represents a 320x50 banner advertisement.
     */
    public static final MaxAdFormat BANNER = new MaxAdFormat( "BANNER" );

    /**
     * Represents a 300x250 rectangular advertisement.
     */
    public static final MaxAdFormat MREC = new MaxAdFormat( "MREC" );

    /**
     * Represents a 728x90 leaderboard advertisement (for tablets).
     */
    public static final MaxAdFormat LEADER = new MaxAdFormat( "LEADER" );

    /**
     * Represents a full-screen advertisement.
     */
    public static final MaxAdFormat INTERSTITIAL = new MaxAdFormat( "INTER" );

    /**
     * Similar to {@link MaxAdFormat#INTERSTITIAL}, except that users are given a reward at the end of the advertisement.
     */
    public static final MaxAdFormat REWARDED = new MaxAdFormat( "REWARDED" );

    /**
     * Represents a fullscreen ad which the user can skip and be granted a reward upon successful completion of the ad.
     */
    public static final MaxAdFormat REWARDED_INTERSTITIAL = new MaxAdFormat( "REWARDED_INTER" );

    /**
     * Represents a native advertisement.
     */
    public static final MaxAdFormat NATIVE = new MaxAdFormat( "NATIVE" );

    private final String label;

    private MaxAdFormat(final String label)
    {
        this.label = label;
    }

    public String getLabel()
    {
        return label;
    }

    /**
     * @return The size of the AdView format ad, or Size with (width: 0, height: 0) otherwise.
     */
    public AppLovinSdkUtils.Size getSize()
    {
        if ( this == BANNER )
        {
            return new AppLovinSdkUtils.Size( 320, 50 );
        }
        else if ( this == LEADER )
        {
            return new AppLovinSdkUtils.Size( 728, 90 );
        }
        else if ( this == MREC )
        {
            return new AppLovinSdkUtils.Size( 300, 250 );
        }
        else
        {
            return new AppLovinSdkUtils.Size( 0, 0 );
        }
    }

    /**
     * Get the adaptive banner size for the screen width at the current orientation.
     * <p>
     * NOTE: The height is currently the only "adaptive" dimension and the width will span the screen.
     * NOTE: Only AdMob / Google Ad Manager currently has support for adaptive banners and the maximum height is 15% the height of the screen.
     */
    public AppLovinSdkUtils.Size getAdaptiveSize(final Activity activity)
    {
        return getAdaptiveSize( -1, activity );
    }

    /**
     * Get the adaptive banner {@code AppLovinSdkUtils#Size} for the provided width at the current orientation.
     * <p>
     * NOTE: The height is currently the only "adaptive" dimension and the provided width will be passed back in the returned {@code AppLovinSdkUtils#Size}.
     * NOTE: Only AdMob / Google Ad Manager currently has support for adaptive banners.
     *
     * @param widthDp  The width in density-independent pixels to retrieve the adaptive banner size for.
     * @param activity The {@link Activity} in which the banner will be shown.
     *
     * @return The adaptive banner size for the current orientation and width.
     */
    public AppLovinSdkUtils.Size getAdaptiveSize(final int widthDp, final Activity activity)
    {
        if ( this == BANNER || this == LEADER )
        {
            return MediationUtils.getAdaptiveBannerAdSize( widthDp, this, activity );
        }
        else
        {
            return getSize();
        }
    }

    /**
     * @return {@code true} if the ad format is an interstitial, rewarded, or rewarded interstitial.
     */
    public boolean isFullscreenAd()
    {
        return this == INTERSTITIAL || this == REWARDED || this == REWARDED_INTERSTITIAL;
    }

    /**
     * @return {@code true} if the ad format is a banner, leader, or MREC.
     */
    public boolean isAdViewAd()
    {
        return this == BANNER || this == MREC || this == LEADER;
    }

    @Override
    public String toString()
    {
        return "MaxAdFormat{" +
                "label='" + label + '\'' +
                '}';
    }
}
