package com.applovin.sdk;

import android.content.Context;

import com.applovin.impl.sdk.AppLovinSdkInitializationConfigurationImpl;

import java.util.List;

import androidx.annotation.Nullable;

/**
 * This class contains configurable properties and settings for the initialization of the AppLovin SDK.
 * To be used in @{@link AppLovinSdk#initialize(AppLovinSdkInitializationConfiguration, AppLovinSdk.SdkInitializationListener)}.
 */
public interface AppLovinSdkInitializationConfiguration
{
    static Builder builder(final String sdkKey, final Context context)
    {
        return new AppLovinSdkInitializationConfigurationImpl.BuilderImpl( sdkKey, context );
    }

    /**
     * @return the SDK key for the AppLovin SDK.
     */
    @Nullable
    String getSdkKey();

    /**
     * @return mediation provider.
     */
    @Nullable
    String getMediationProvider();

    /**
     * @return the plugin version for the mediation adapter or plugin.
     */
    @Nullable
    String getPluginVersion();

    /**
     * @return {@link AppLovinSdkSettings} object for the AppLovinSDK.
     */
    AppLovinSdkSettings getSettings();

    /**
     * @return user segment.
     */
    @Nullable
    AppLovinUserSegment getUserSegment();

    /**
     * @return targeting data.
     */
    @Nullable
    AppLovinTargetingData getTargetingData();

    /**
     * @return the list of advertising identifiers that will receive test ads.
     */
    List<String> getTestDeviceAdvertisingIds();

    /**
     * @return the list of MAX ad unit ids that will be used for this instance of the SDK.
     */
    List<String> getAdUnitIds();

    /**
     * @return whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
     */
    boolean isExceptionHandlerEnabled();

    /**
     * Interface used to configure the {@link AppLovinSdkSettings} object in the Builder {@link AppLovinSdkInitializationConfiguration.Builder#configureSettings(SettingsConfigurator)}
     */
    interface SettingsConfigurator
    {
        void configure(final AppLovinSdkSettings settings);
    }

    /**
     * Builder class used to create an {@link AppLovinSdkInitializationConfiguration} object.
     * This class contains configurable properties and settings for the initialization of the AppLovin SDK.
     */
    interface Builder
    {
        /**
         * Set mediation provider using one of the provided strings above specified by {@link AppLovinMediationProvider}, or your own if not defined.
         */
        Builder setMediationProvider(@Nullable final String mediationProvider);

        /**
         * Sets the plugin version for the mediation adapter or plugin.
         */
        Builder setPluginVersion(@Nullable final String version);

        /**
         * Configure settings object for the AppLovinSDK to initialize with. Note: its properties are mutable and may be modified after initialization.
         */
        Builder configureSettings(@Nullable final SettingsConfigurator settingsConfigurator);

        /**
         * A user segment allows AppLovin to serve ads by using custom-defined rules that are based on which segment the user is in. The user segment is a custom string
         * of 32 alphanumeric characters or less.
         */
        Builder setUserSegment(@Nullable final AppLovinUserSegment userSegment);

        /**
         * A class which allows you to send any demographic or interest-based targeting data.
         */
        Builder setTargetingData(@Nullable final AppLovinTargetingData targetingData);

        /**
         * Enable devices to receive test ads, by passing in the advertising identifier (GAID or App Set ID) of each test device.
         * Refer to AppLovin logs for the GAID or App Set ID of your current device.
         */
        Builder setTestDeviceAdvertisingIds(final List<String> testDeviceAdvertisingIds);

        /**
         * Set the MAX ad unit ids that will be used for this instance of the SDK. 3rd-party SDKs will be initialized with the credentials configured for these ad unit ids.
         */
        Builder setAdUnitIds(final List<String> adUnitIds);

        /**
         * Set whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
         */
        Builder setExceptionHandlerEnabled(final boolean exceptionHandlerEnabled);

        /**
         * Build method to create an instance of {@link AppLovinSdkInitializationConfiguration}.
         *
         * @return an instance of {@link AppLovinSdkInitializationConfiguration}.
         */
        AppLovinSdkInitializationConfiguration build();

        /**
         * @return the SDK key for the AppLovin SDK.
         */
        String getSdkKey();

        /**
         * @return mediation provider.
         */
        @Nullable
        String getMediationProvider();

        /**
         * @return the plugin version for the mediation adapter or plugin.
         */
        @Nullable
        String getPluginVersion();

        /**
         * @return {@link AppLovinSdkSettings} object for the AppLovinSDK.
         */
        AppLovinSdkSettings getSettings();

        /**
         * @return user segment.
         */
        @Nullable
        AppLovinUserSegment getUserSegment();

        /**
         * @return targeting data.
         */
        @Nullable
        AppLovinTargetingData getTargetingData();

        /**
         * @return the list of advertising identifiers that will receive test ads.
         */
        List<String> getTestDeviceAdvertisingIds();

        /**
         * @return the list of MAX ad unit ids that will be used for this instance of the SDK.
         */
        List<String> getAdUnitIds();

        /**
         * @return whether or not the AppLovin SDK listens to exceptions. Defaults to {@code true}.
         */
        boolean isExceptionHandlerEnabled();
    }
}
