package com.applovin.mediation.hybridAds;

import android.view.View;

import com.applovin.impl.mediation.model.MediatedFullscreenAd;
import com.applovin.impl.sdk.CoreSdk;
import com.applovin.impl.sdk.utils.ActivityUtils;
import com.applovin.impl.sdk.utils.Utils;
import com.applovin.mediation.adapter.MaxAdapterError;
import com.applovin.mediation.adapter.listeners.MaxAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxAppOpenAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener;
import com.applovin.mediation.nativeAds.MaxNativeAd;

import androidx.annotation.Nullable;
import lombok.Setter;
import lombok.val;

public class MaxHybridAdService
{
    private static final String TAG = "MaxHybridAdService";

    private final CoreSdk sdk;

    @Setter
    @Nullable
    private MediatedFullscreenAd pendingFullscreenAd;

    public MaxHybridAdService(final CoreSdk sdk)
    {
        this.sdk = sdk;
    }

    //region API

    /**
     * Show a fullscreen ad for the provided MREC using MAX. This only supports interstitial and app open ads.
     *
     * @param mrec     The MREC {@code View} to display as a fullscreen ad.
     * @param listener The adapter's listener. Traditionally this is abstracted away in each adapter, but it is consolidated for hybrid ads.
     */
    public void showFullscreenAd(final View mrec, final MaxAdapterListener listener)
    {
        Utils.assertMainThread();

        if ( pendingFullscreenAd == null )
        {
            sdk.getLogger().e( TAG, "Failed to display hybrid MREC ad: pending ad is null" );

            if ( listener instanceof MaxInterstitialAdapterListener )
            {
                ( (MaxInterstitialAdapterListener) listener ).onInterstitialAdDisplayFailed( MaxAdapterError.AD_DISPLAY_FAILED );
            }
            else if ( listener instanceof MaxAppOpenAdapterListener )
            {
                ( (MaxAppOpenAdapterListener) listener ).onAppOpenAdDisplayFailed( MaxAdapterError.AD_DISPLAY_FAILED );
            }

            return;
        }

        val userActivity = sdk.getActivityLifecycleManager().getTopActivity();
        ActivityUtils.startActivity( userActivity,
                                     MaxHybridMRecAdActivity.class,
                                     sdk.getActivityLifecycleManager(),
                                     new ActivityUtils.ActivityInitializer<MaxHybridMRecAdActivity>()
                                     {
                                         @Override
                                         public void onActivityCreated(final MaxHybridMRecAdActivity activity)
                                         {
                                             activity.initialize( mrec, sdk, listener );
                                         }
                                     } );
    }

    /**
     * Show a fullscreen ad for the provided native ad using MAX. This only supports interstitial and app open ads.
     *
     * @param nativeAd The native ad to display the fullscreen ad for.
     * @param listener The adapter's listener. Traditionally this is abstracted away in each adapter, but it is consolidated for hybrid ads.
     */
    public void showFullscreenAd(final MaxNativeAd nativeAd, final MaxAdapterListener listener)
    {
        Utils.assertMainThread();

        if ( pendingFullscreenAd == null )
        {
            sdk.getLogger().e( TAG, "Failed to display hybrid native ad: pending ad is null" );

            if ( listener instanceof MaxInterstitialAdapterListener )
            {
                ( (MaxInterstitialAdapterListener) listener ).onInterstitialAdDisplayFailed( MaxAdapterError.AD_DISPLAY_FAILED );
            }
            else if ( listener instanceof MaxAppOpenAdapterListener )
            {
                ( (MaxAppOpenAdapterListener) listener ).onAppOpenAdDisplayFailed( MaxAdapterError.AD_DISPLAY_FAILED );
            }

            return;
        }

        val userActivity = sdk.getActivityLifecycleManager().getTopActivity();
        ActivityUtils.startActivity( userActivity,
                                     MaxHybridNativeAdActivity.class,
                                     sdk.getActivityLifecycleManager(),
                                     new ActivityUtils.ActivityInitializer<MaxHybridNativeAdActivity>()
                                     {
                                         @Override
                                         public void onActivityCreated(final MaxHybridNativeAdActivity activity)
                                         {
                                             activity.initialize( nativeAd, sdk, listener );
                                         }
                                     } );
    }

    /**
     * @return the pending fullscreen ad and clear it.
     */
    protected MediatedFullscreenAd getAndClearPendingFullscreenAd()
    {
        val pendingAd = pendingFullscreenAd;
        pendingFullscreenAd = null;

        return pendingAd;
    }

    //endregion
}
