package com.applovin.adview;

import android.content.Context;

import com.applovin.impl.adview.InterstitialAdDialogWrapper;
import com.applovin.impl.sdk.Logger;
import com.applovin.sdk.AppLovinAd;
import com.applovin.sdk.AppLovinAdSize;
import com.applovin.sdk.AppLovinSdk;

import java.lang.ref.WeakReference;

/**
 * This class represents an interstitial ad that is rendered on top of the current activity.
 *
 * @author Basil Shikin
 */
public class AppLovinInterstitialAd
{
    private static final String TAG = "AppLovinInterstitialAd";

    private static final Object                      lastCreatedDialogLock = new Object();
    private static       InterstitialAdDialogWrapper lastCreatedDialog;

    private static WeakReference<Context> lastContextRef = new WeakReference<>( null );

    /**
     * Create a new interstitial dialog that can be shown to the user. This is primarily useful in advanced integrations as the Ad Dialog allows finer control - including manually pre-caching and rendering ads. If all you want to do is show an interstitial the default way, you're looking for
     * <code>show()</code>.
     *
     * @param sdk     An SDK instance to use.
     * @param context A non-stale reference to the current top activity.
     *
     * @return A new instance of {@link com.applovin.adview.AppLovinInterstitialAdDialog}.
     */
    public static AppLovinInterstitialAdDialog create(final AppLovinSdk sdk, final Context context)
    {
        // Check input
        if ( sdk == null ) throw new IllegalArgumentException( "No sdk specified" );
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        return createDialog( sdk, context );
    }

    private static InterstitialAdDialogWrapper createDialog(final AppLovinSdk sdk, final Context context)
    {
        synchronized ( lastCreatedDialogLock )
        {
            lastCreatedDialog = new InterstitialAdDialogWrapper( sdk, context );
            lastContextRef = new WeakReference<>( context );
        }

        return lastCreatedDialog;
    }

    // TODO: Remove this once we remove AppLovinInterstitialAd.show()
    private static InterstitialAdDialogWrapper getLastCreatedDialog(final AppLovinSdk sdk, final Context context)
    {
        synchronized ( lastCreatedDialogLock )
        {
            if ( lastCreatedDialog == null || lastContextRef.get() != context )
            {
                Logger.userWarning( TAG, "No interstitial dialog was previously created or was created with a different context. Creating new interstitial dialog" );
                lastCreatedDialog = new InterstitialAdDialogWrapper( sdk, context );
                lastContextRef = new WeakReference<>( context );
            }
        }

        return lastCreatedDialog;
    }

    /**
     * @deprecated Statically displaying an ad has been deprecated and will be removed in a future SDK version. Please create an instance of {@link AppLovinInterstitialAdDialog} and use {@link AppLovinInterstitialAdDialog#show()} or {@link AppLovinInterstitialAdDialog#showAndRender(AppLovinAd)} to display an ad.
     */
    @Deprecated
    public static void show(final Context context)
    {
        // Check input
        if ( context == null ) throw new IllegalArgumentException( "No context specified" );

        getLastCreatedDialog( AppLovinSdk.getInstance( context ), context ).show();
    }

    /**
     * @deprecated Checking whether an ad is ready for display has been deprecated and will be removed in a future SDK version. Please create an instance of {@link AppLovinInterstitialAdDialog} and use {@link AppLovinInterstitialAdDialog#show()} or {@link AppLovinInterstitialAdDialog#showAndRender(AppLovinAd)} to display an ad.
     */
    @Deprecated
    public static boolean isAdReadyToDisplay(final Context context)
    {
        return AppLovinSdk.getInstance( context ).getAdService().hasPreloadedAd( AppLovinAdSize.INTERSTITIAL );
    }

    @Override
    public String toString()
    {
        return "AppLovinInterstitialAd{}";
    }
}
