/*
 * Decompiled with CFR 0.152.
 */
package com.applitools.utils;

import com.applitools.eyes.EyesException;
import com.applitools.eyes.Region;
import com.applitools.eyes.ScaleProvider;
import com.applitools.utils.ArgumentGuard;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.BufferedImageOp;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.Raster;
import java.awt.image.RenderedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import javax.imageio.ImageIO;
import org.apache.commons.codec.binary.Base64;
import org.imgscalr.Scalr;

public class ImageUtils {
    public static final int REQUIRED_IMAGE_TYPE = 6;

    public static BufferedImage normalizeImageType(BufferedImage image) {
        if (image.getType() == 6) {
            return image;
        }
        return ImageUtils.copyImageWithType(image, 6);
    }

    public static byte[] encodeAsPng(BufferedImage image) {
        byte[] encodedImage;
        ArgumentGuard.notNull(image, "image");
        ByteArrayOutputStream pngBytesStream = new ByteArrayOutputStream();
        try {
            ImageIO.write((RenderedImage)image, "png", pngBytesStream);
            pngBytesStream.flush();
            encodedImage = pngBytesStream.toByteArray();
        }
        catch (IOException e) {
            throw new EyesException("Failed to encode image", e);
        }
        finally {
            try {
                pngBytesStream.close();
            }
            catch (IOException e) {
                throw new EyesException("Failed to close png byte stream", e);
            }
        }
        return encodedImage;
    }

    public static BufferedImage imageFromFile(String path) throws EyesException {
        BufferedImage image;
        try {
            image = ImageIO.read(new File(path));
            image = ImageUtils.normalizeImageType(image);
        }
        catch (IOException e) {
            throw new EyesException("Failed to to load the image bytes from " + path, e);
        }
        return image;
    }

    public static BufferedImage imageFromResource(String resource) throws EyesException {
        BufferedImage image;
        try {
            image = ImageIO.read(ImageUtils.class.getClassLoader().getResourceAsStream(resource));
            image = ImageUtils.normalizeImageType(image);
        }
        catch (IOException e) {
            throw new EyesException("Failed to to load the image from resource: " + resource, e);
        }
        return image;
    }

    public static BufferedImage imageFromBase64(String image64) throws EyesException {
        ArgumentGuard.notNullOrEmpty(image64, "image64");
        byte[] imageBytes = Base64.decodeBase64((byte[])image64.getBytes(Charset.forName("UTF-8")));
        return ImageUtils.imageFromBytes(imageBytes);
    }

    public static String base64FromImage(BufferedImage image) {
        ArgumentGuard.notNull(image, "image");
        byte[] imageBytes = ImageUtils.encodeAsPng(image);
        return Base64.encodeBase64String((byte[])imageBytes);
    }

    public static BufferedImage imageFromBytes(byte[] imageBytes) throws EyesException {
        BufferedImage image;
        try {
            ByteArrayInputStream screenshotStream = new ByteArrayInputStream(imageBytes);
            image = ImageIO.read(screenshotStream);
            screenshotStream.close();
            image = ImageUtils.normalizeImageType(image);
        }
        catch (IOException e) {
            throw new EyesException("Failed to create buffered image!", e);
        }
        return image;
    }

    public static BufferedImage getImagePart(BufferedImage image, Region region) {
        ArgumentGuard.notNull(image, "image");
        BufferedImage imagePart = image.getSubimage(region.getLeft(), region.getTop(), region.getWidth(), region.getHeight());
        byte[] imagePartBytes = ImageUtils.encodeAsPng(imagePart);
        return ImageUtils.imageFromBytes(imagePartBytes);
    }

    public static BufferedImage rotateImage(BufferedImage image, double deg) {
        ArgumentGuard.notNull(image, "image");
        if (deg % 360.0 == 0.0) {
            return image;
        }
        double radians = Math.toRadians(deg);
        double angleSin = Math.abs(Math.sin(radians));
        double angleCos = Math.abs(Math.cos(radians));
        int originalWidth = image.getWidth();
        double originalHeight = image.getHeight();
        int rotatedWidth = (int)Math.floor((double)originalWidth * angleCos + originalHeight * angleSin);
        int rotatedHeight = (int)Math.floor(originalHeight * angleCos + (double)originalWidth * angleSin);
        BufferedImage rotatedImage = new BufferedImage(rotatedWidth, rotatedHeight, image.getType());
        Graphics2D g = rotatedImage.createGraphics();
        g.translate((double)((rotatedWidth - originalWidth) / 2), ((double)rotatedHeight - originalHeight) / 2.0);
        g.rotate(radians, originalWidth / 2, originalHeight / 2.0);
        g.drawRenderedImage(image, null);
        g.dispose();
        return ImageUtils.normalizeImageType(rotatedImage);
    }

    public static BufferedImage copyImageWithType(BufferedImage src, int updatedType) {
        ArgumentGuard.notNull(src, "src");
        BufferedImage result = new BufferedImage(src.getWidth(), src.getHeight(), updatedType);
        Graphics2D g2 = result.createGraphics();
        g2.drawRenderedImage(src, null);
        g2.dispose();
        return result;
    }

    public static BufferedImage scaleImage(BufferedImage image, ScaleProvider scaleProvider) {
        ArgumentGuard.notNull(image, "image");
        ArgumentGuard.notNull(scaleProvider, "scaleProvider");
        double scaleRatio = scaleProvider.getScaleRatio();
        return ImageUtils.scaleImage(image, scaleRatio);
    }

    public static BufferedImage scaleImage(BufferedImage image, double scaleRatio) {
        ArgumentGuard.notNull(image, "image");
        ArgumentGuard.notNull(scaleRatio, "scaleRatio");
        image = ImageUtils.normalizeImageType(image);
        if (scaleRatio == 1.0) {
            return image;
        }
        double imageRatio = (double)image.getHeight() / (double)image.getWidth();
        int scaledWidth = (int)Math.ceil((double)image.getWidth() * scaleRatio);
        int scaledHeight = (int)Math.ceil((double)scaledWidth * imageRatio);
        BufferedImage scaledImage = ImageUtils.resizeImage(image, scaledWidth, scaledHeight);
        return ImageUtils.normalizeImageType(scaledImage);
    }

    public static BufferedImage resizeImage(BufferedImage image, int targetWidth, int targetHeight) {
        ArgumentGuard.notNull(image, "image");
        ArgumentGuard.notNull(targetWidth, "targetWidth");
        ArgumentGuard.notNull(targetHeight, "targetHeight");
        image = ImageUtils.normalizeImageType(image);
        if (image.getWidth() == targetWidth && image.getHeight() == targetHeight) {
            return image;
        }
        BufferedImage resizedImage = targetWidth > image.getWidth() || targetHeight > image.getHeight() ? ImageUtils.scaleImageBicubic(image, targetWidth, targetHeight) : ImageUtils.scaleImageIncrementally(image, targetWidth, targetHeight);
        return ImageUtils.normalizeImageType(resizedImage);
    }

    private static int interpolateCubic(int x0, int x1, int x2, int x3, double t) {
        int a0 = x3 - x2 - x0 + x1;
        int a1 = x0 - x1 - a0;
        int a2 = x2 - x0;
        return (int)Math.max(0.0, Math.min(255.0, (double)a0 * (t * t * t) + (double)a1 * (t * t) + (double)a2 * t + (double)x1));
    }

    private static BufferedImage scaleImageBicubic(BufferedImage srcImage, int targetWidth, int targetHeight) {
        int yPos;
        double y;
        int kPos;
        int k;
        int buf1Pos;
        double t;
        int xPos;
        double x;
        int j;
        int i;
        ImageUtils.normalizeImageType(srcImage);
        DataBuffer bufSrc = srcImage.getRaster().getDataBuffer();
        DataBufferByte bufDst = new DataBufferByte(targetWidth * targetHeight * 4);
        int wSrc = srcImage.getWidth();
        int hSrc = srcImage.getHeight();
        int wM = (int)Math.max(1.0, Math.floor(wSrc / targetWidth));
        int wDst2 = targetWidth * wM;
        int hM = (int)Math.max(1.0, Math.floor(hSrc / targetHeight));
        int hDst2 = targetHeight * hM;
        DataBufferByte buf1 = new DataBufferByte(wDst2 * hSrc * 4);
        for (i = 0; i < hSrc; ++i) {
            for (j = 0; j < wDst2; ++j) {
                x = (double)j * (double)(wSrc - 1) / (double)wDst2;
                xPos = (int)Math.floor(x);
                t = x - (double)xPos;
                int srcPos = (i * wSrc + xPos) * 4;
                buf1Pos = (i * wDst2 + j) * 4;
                for (k = 0; k < 4; ++k) {
                    kPos = srcPos + k;
                    int x0 = xPos > 0 ? bufSrc.getElem(kPos - 4) : 2 * bufSrc.getElem(kPos) - bufSrc.getElem(kPos + 4);
                    int x1 = bufSrc.getElem(kPos);
                    int x2 = bufSrc.getElem(kPos + 4);
                    int x3 = xPos < wSrc - 2 ? bufSrc.getElem(kPos + 8) : 2 * bufSrc.getElem(kPos + 4) - bufSrc.getElem(kPos);
                    ((DataBuffer)buf1).setElem(buf1Pos + k, ImageUtils.interpolateCubic(x0, x1, x2, x3, t));
                }
            }
        }
        DataBufferByte buf2 = new DataBufferByte(wDst2 * hDst2 * 4);
        for (i = 0; i < hDst2; ++i) {
            for (j = 0; j < wDst2; ++j) {
                y = (double)i * (double)(hSrc - 1) / (double)hDst2;
                yPos = (int)Math.floor(y);
                t = y - (double)yPos;
                buf1Pos = (yPos * wDst2 + j) * 4;
                int buf2Pos = (i * wDst2 + j) * 4;
                for (k = 0; k < 4; ++k) {
                    kPos = buf1Pos + k;
                    int y0 = yPos > 0 ? ((DataBuffer)buf1).getElem(kPos - wDst2 * 4) : 2 * ((DataBuffer)buf1).getElem(kPos) - ((DataBuffer)buf1).getElem(kPos + wDst2 * 4);
                    int y1 = ((DataBuffer)buf1).getElem(kPos);
                    int y2 = ((DataBuffer)buf1).getElem(kPos + wDst2 * 4);
                    int y3 = yPos < hSrc - 2 ? ((DataBuffer)buf1).getElem(kPos + wDst2 * 8) : 2 * ((DataBuffer)buf1).getElem(kPos + wDst2 * 4) - ((DataBuffer)buf1).getElem(kPos);
                    ((DataBuffer)buf2).setElem(buf2Pos + k, ImageUtils.interpolateCubic(y0, y1, y2, y3, t));
                }
            }
        }
        int m = wM * hM;
        if (m > 1) {
            for (i = 0; i < targetHeight; ++i) {
                for (j = 0; j < targetWidth; ++j) {
                    int r = 0;
                    int g = 0;
                    int b = 0;
                    int a = 0;
                    for (y = 0.0; y < (double)hM; y += 1.0) {
                        yPos = (int)((double)(i * hM) + y);
                        for (x = 0.0; x < (double)wM; x += 1.0) {
                            xPos = (int)((double)(j * wM) + x);
                            int xyPos = (yPos * wDst2 + xPos) * 4;
                            r += ((DataBuffer)buf2).getElem(xyPos);
                            g += ((DataBuffer)buf2).getElem(xyPos + 1);
                            b += ((DataBuffer)buf2).getElem(xyPos + 2);
                            a += ((DataBuffer)buf2).getElem(xyPos + 3);
                        }
                    }
                    int pos = (i * targetWidth + j) * 4;
                    ((DataBuffer)bufDst).setElem(pos, Math.round(r / m));
                    ((DataBuffer)bufDst).setElem(pos + 1, Math.round(g / m));
                    ((DataBuffer)bufDst).setElem(pos + 2, Math.round(b / m));
                    ((DataBuffer)bufDst).setElem(pos + 3, Math.round(a / m));
                }
            }
        } else {
            bufDst = buf2;
        }
        BufferedImage dstImage = new BufferedImage(targetWidth, targetHeight, 6);
        dstImage.setData(Raster.createRaster(dstImage.getSampleModel(), bufDst, null));
        return dstImage;
    }

    private static BufferedImage scaleImageIncrementally(BufferedImage src, int targetWidth, int targetHeight) {
        boolean hasReassignedSrc = false;
        src = ImageUtils.normalizeImageType(src);
        int currentWidth = src.getWidth();
        int currentHeight = src.getHeight();
        int fraction = 2;
        do {
            int prevCurrentWidth = currentWidth;
            int prevCurrentHeight = currentHeight;
            if (currentWidth > targetWidth && (currentWidth -= currentWidth / fraction) < targetWidth) {
                currentWidth = targetWidth;
            }
            if (currentHeight > targetHeight && (currentHeight -= currentHeight / fraction) < targetHeight) {
                currentHeight = targetHeight;
            }
            if (prevCurrentWidth == currentWidth && prevCurrentHeight == currentHeight) break;
            BufferedImage incrementalImage = ImageUtils.scaleImageBicubic(src, currentWidth, currentHeight);
            if (hasReassignedSrc) {
                src.flush();
            }
            src = incrementalImage;
            hasReassignedSrc = true;
        } while (currentWidth != targetWidth || currentHeight != targetHeight);
        return src;
    }

    public static BufferedImage cropImage(BufferedImage image, Region regionToCrop) {
        BufferedImage croppedImage = Scalr.crop((BufferedImage)image, (int)regionToCrop.getLeft(), (int)regionToCrop.getTop(), (int)regionToCrop.getWidth(), (int)regionToCrop.getHeight(), (BufferedImageOp[])new BufferedImageOp[0]);
        return ImageUtils.normalizeImageType(croppedImage);
    }

    public static void saveImage(BufferedImage image, String filename) {
        try {
            File file = new File(filename);
            ImageIO.write((RenderedImage)image, "png", file);
        }
        catch (IOException e) {
            throw new EyesException("Failed to save image", e);
        }
    }
}

