package com.aol.cyclops.data.collections.extensions.persistent;

import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Optional;
import java.util.Random;
import java.util.function.BiFunction;
import java.util.function.BiPredicate;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.stream.Collector;
import java.util.stream.Stream;

import org.jooq.lambda.Seq;
import org.jooq.lambda.tuple.Tuple2;
import org.jooq.lambda.tuple.Tuple3;
import org.jooq.lambda.tuple.Tuple4;
import org.pcollections.PVector;
import org.pcollections.TreePVector;
import org.reactivestreams.Publisher;

import com.aol.cyclops.Monoid;
import com.aol.cyclops.Reducer;
import com.aol.cyclops.Reducers;
import com.aol.cyclops.control.Matchable.CheckValue1;
import com.aol.cyclops.control.ReactiveSeq;
import com.aol.cyclops.control.Trampoline;
import com.aol.cyclops.data.collections.extensions.standard.ListX;

public interface PVectorX<T> extends PVector<T>, PersistentCollectionX<T>{
	
    /**
     * Create a PVectorX that contains the Integers between start and end
     * 
     * @param start
     *            Number of range to start from
     * @param end
     *            Number for range to end at
     * @return Range PVectorX
     */
    public static PVectorX<Integer> range(int start, int end) {
        return ReactiveSeq.range(start,end).toPVectorX();
    }
    /**
     * Create a PVectorX that contains the Longs between start and end
     * 
     * @param start
     *            Number of range to start from
     * @param end
     *            Number for range to end at
     * @return Range PVectorX
     */
    public static PVectorX<Long> rangeLong(long start, long end) {
        return ReactiveSeq.rangeLong(start,end).toPVectorX();
    }
    /**
     * Unfold a function into a PVectorX
     * 
     * <pre>
     * {@code 
     *  PVectorX.unfold(1,i->i<=6 ? Optional.of(Tuple.tuple(i,i+1)) : Optional.empty());
     * 
     * //(1,2,3,4,5)
     * 
     * }</code>
     * 
     * @param seed Initial value 
     * @param unfolder Iteratively applied function, terminated by an empty Optional
     * @return PVectorX generated by unfolder function
     */
    static <U,T> PVectorX<T> unfold(U seed,Function<? super U,Optional<Tuple2<T,U>>> unfolder){
        return ReactiveSeq.unfold(seed, unfolder).toPVectorX();
    }
    /**
     * Generate a PVectorX from the provided Supplier up to the provided limit number of times
     * 
     * @param limit Max number of elements to generate
     * @param s Supplier to generate PVectorX elements
     * @return PVectorX generated from the provided Supplier
     */
    public static <T> PVectorX<T> generate(long limit,Supplier<T> s){

        return ReactiveSeq.generate(s).limit(limit).toPVectorX();
     }
    /**
     * Create a PVectorX by iterative application of a function to an initial element up to the supplied limit number of times
     * 
     * @param limit Max number of elements to generate
     * @param seed Initial element
     * @param f Iteratively applied to each element to generate the next element
     * @return PVectorX generated by iterative application
     */
    public static <T> PVectorX<T> iterate(long limit, final T seed, final UnaryOperator<T> f) {
        return ReactiveSeq.iterate(seed, f).limit(limit).toPVectorX();
        
    }
    
	/**
	 * Construct a PVector from the provided values 
	 * 
	 * <pre>
	 * {@code 
	 *  List<String> list = PVectors.of("a","b","c");
	 *  
	 *  // or
	 *  
	 *  PVector<String> list = PVectors.of("a","b","c");
	 *  
	 *  
	 * }
	 * </pre>
	 * 
	 * 
	 * @param values To add to PVector
	 * @return new PVector
	 */
	public static <T> PVectorX<T> of(T...values){
		return new PVectorXImpl<>(TreePVector.from(Arrays.asList(values)));
	}
	/**
	 * <pre>
	 * {@code 
	 *     List<String> empty = PVectors.empty();
	 *    //or
	 *    
	 *     PVector<String> empty = PVectors.empty();
	 * }
	 * </pre>
	 * @return an empty PVector
	 */
	public static<T> PVectorX<T> empty(){
		return new PVectorXImpl<>(TreePVector .empty());
	}
	/**
	 * Construct a PVector containing a single value
	 * </pre>
	 * {@code 
	 *    List<String> single = PVectors.singleton("1");
	 *    
	 *    //or
	 *    
	 *    PVector<String> single = PVectors.singleton("1");
	 * 
	 * }
	 * </pre>
	 * 
	 * @param value Single value for PVector
	 * @return PVector with a single value
	 */
	public static <T> PVectorX<T> singleton(T value){
		return new PVectorXImpl<>(TreePVector.singleton(value));
	}
    /**
     * Construct a PVectorX from an Publisher
     * 
     * @param publisher
     *            to construct PVectorX from
     * @return PVectorX
     */
    public static <T> PVectorX<T> fromPublisher(Publisher<? extends T> publisher) {
        return ReactiveSeq.fromPublisher((Publisher<T>)publisher).toPVectorX();
    }
	public static<T> PVectorX<T> fromIterable(Iterable<T> iterable){
		if(iterable instanceof PVectorX)
			return (PVectorX)iterable;
		if(iterable instanceof PVector)
			return new PVectorXImpl<>((PVector)(iterable));
		PVector<T> res = TreePVector.<T>empty();
		Iterator<T> it = iterable.iterator();
		while(it.hasNext())
			res = res.plus(it.next());
		
		return new PVectorXImpl<>(res);
	}
	/**
	 * Create a PVector from the supplied Colleciton
	 * <pre>
	 * {@code 
	 *   PVector<Integer> list = PVectors.fromCollection(Arrays.asList(1,2,3));
	 *   
	 * }
	 * </pre>
	 * 
	 * @param values to add to new PVector
	 * @return PVector containing values
	 */
	public static <T> PVectorX<T> fromCollection(Collection<T> values){
		if(values instanceof PVectorX)
			return (PVectorX)values;
		if(values instanceof PVector)
			return new PVectorXImpl<>((PVector)values);
		return new PVectorXImpl<>(TreePVector.from(values));
	}
	/**
	 * Reduce (immutable Collection) a Stream to a PVector
	 * 
	 * <pre>
	 * {@code 
	 *    PVector<Integer> list = PVectors.fromStream(Stream.of(1,2,3));
	 * 
	 *  //list = [1,2,3]
	 * }</pre>
	 * 
	 * 
	 * @param stream to convert to a PVector
	 * @return
	 */
	public static<T> PVectorX<T> fromStream(Stream<T> stream){
		return Reducers.<T>toPVectorX().mapReduce(stream);
	}
	  /**
     * Combine two adjacent elements in a PVectorX using the supplied BinaryOperator
     * This is a stateful grouping & reduction operation. The output of a combination may in turn be combined
     * with it's neighbor
     * <pre>
     * {@code 
     *  PVectorX.of(1,1,2,3)
                   .combine((a, b)->a.equals(b),Semigroups.intSum)
                   .toListX()
                   
     *  //ListX(3,4) 
     * }</pre>
     * 
     * @param predicate Test to see if two neighbors should be joined
     * @param op Reducer to combine neighbors
     * @return Combined / Partially Reduced PVectorX
     */
    default PVectorX<T> combine(BiPredicate<? super T, ? super T> predicate, BinaryOperator<T> op){
        return (PVectorX<T>)PersistentCollectionX.super.combine(predicate,op);
    }
	
	default PVector<T> toPVector(){
		return this;
	}
	
	default <X> PVectorX<X> from(Collection<X> col){
		return fromCollection(col);
	}
	default <T> Reducer<PVector<T>> monoid(){
		return Reducers.toPVector();
	}
	@Override
	default PVectorX<T> toPVectorX() {
		return this;
	}
	
	

	

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#reverse()
	 */
	@Override
	default PVectorX<T> reverse() {
		return (PVectorX<T>)PersistentCollectionX.super.reverse();
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#filter(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> filter(Predicate<? super T> pred) {
		return (PVectorX<T>)PersistentCollectionX.super.filter(pred);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#map(java.util.function.Function)
	 */
	@Override
	default <R> PVectorX<R> map(Function<? super T, ? extends R> mapper) {
		
		return (PVectorX<R>)PersistentCollectionX.super.map(mapper);
	}
	@Override
	default<R> PVectorX<R> unit(Collection<R> col){
		return fromCollection(col);
	}
	@Override
	default <R> PVectorX<R> unit(R value){
		return singleton(value);
	}
	@Override
	default<R> PVectorX<R> emptyUnit(){
		return empty();
	}
	@Override
	default <R> PVectorX<R> unitIterator(Iterator<R> it){
		return fromIterable(()->it);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#flatMap(java.util.function.Function)
	 */
	@Override
	default <R> PVectorX<R> flatMap(Function<? super T, ? extends Iterable<? extends R>> mapper) {
		
		return (PVectorX<R>)PersistentCollectionX.super.flatMap(mapper);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#limit(long)
	 */
	@Override
	default PVectorX<T> limit(long num) {
		return (PVectorX<T>)PersistentCollectionX.super.limit(num);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#skip(long)
	 */
	@Override
	default PVectorX<T> skip(long num) {
		return (PVectorX<T>)PersistentCollectionX.super.skip(num);
	}
	default PVectorX<T> takeRight(int num){
		return (PVectorX<T>)PersistentCollectionX.super.takeRight(num);
	}
	default PVectorX<T> dropRight(int num){
		return (PVectorX<T>)PersistentCollectionX.super.dropRight(num);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#takeWhile(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> takeWhile(Predicate<? super T> p) {
		return (PVectorX<T>)PersistentCollectionX.super.takeWhile(p);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#dropWhile(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> dropWhile(Predicate<? super T> p) {
		return (PVectorX<T>)PersistentCollectionX.super.dropWhile(p);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#takeUntil(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> takeUntil(Predicate<? super T> p) {
		return (PVectorX<T>)PersistentCollectionX.super.takeUntil(p);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#dropUntil(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> dropUntil(Predicate<? super T> p) {
		return (PVectorX<T>)PersistentCollectionX.super.dropUntil(p);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#trampoline(java.util.function.Function)
	 */
	@Override
	default <R> PVectorX<R> trampoline(Function<? super T, ? extends Trampoline<? extends R>> mapper) {
		return (PVectorX<R>)PersistentCollectionX.super.trampoline(mapper);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#slice(long, long)
	 */
	@Override
	default PVectorX<T> slice(long from, long to) {
		return (PVectorX<T>)PersistentCollectionX.super.slice(from, to);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#sorted(java.util.function.Function)
	 */
	@Override
	default <U extends Comparable<? super U>> PVectorX<T> sorted(Function<? super T, ? extends U> function) {
		return (PVectorX<T>)PersistentCollectionX.super.sorted(function);
	}
	
	public PVectorX<T> plus(T e);
	
	
	public PVectorX<T> plusAll(Collection<? extends T> list);
	
	public PVectorX<T> with(int i, T e);
	
	
	public PVectorX<T> plus(int i, T e);
	
	
	public PVectorX<T> plusAll(int i, Collection<? extends T> list);
	
	
	public PVectorX<T> minus(Object e);
	
	
	public PVectorX<T> minusAll(Collection<?> list);
	
	
	public PVectorX<T> minus(int i);

	
	public PVectorX<T> subList(int start, int end);
	
	default PVectorX<ListX<T>> grouped(int groupSize){
		return  (PVectorX<ListX<T>>)PersistentCollectionX.super.grouped(groupSize);
	}
	default <K, A, D> PVectorX<Tuple2<K, D>> grouped(Function<? super T, ? extends K> classifier, Collector<? super T, A, D> downstream){
		return  (PVectorX)PersistentCollectionX.super.grouped(classifier,downstream);
	}
	default <K> PVectorX<Tuple2<K, Seq<T>>> grouped(Function<? super T, ? extends K> classifier){
		return  (PVectorX)PersistentCollectionX.super.grouped(classifier);
	}
	default <U> PVectorX<Tuple2<T, U>> zip(Iterable<? extends U> other){
		return  (PVectorX)PersistentCollectionX.super.zip(other);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zip(java.lang.Iterable, java.util.function.BiFunction)
	 */
	@Override
	default <U, R> PVectorX<R> zip(Iterable<? extends U> other,
			BiFunction<? super T, ? super U, ? extends R> zipper) {
		
		return (PVectorX<R>)PersistentCollectionX.super.zip(other, zipper);
	}
	@Override
    default <U, R> PVectorX<R> zip(Seq<? extends U> other,
            BiFunction<? super T, ? super U, ? extends R> zipper) {
        
        return (PVectorX<R>)PersistentCollectionX.super.zip(other, zipper);
    }

	@Override
    default <U, R> PVectorX<R> zip(Stream<? extends U> other,
            BiFunction<? super T, ? super U, ? extends R> zipper) {
        
        return (PVectorX<R>)PersistentCollectionX.super.zip(other, zipper);
    }
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#permutations()
	 */
	@Override
	default PVectorX<ReactiveSeq<T>> permutations() {
		
		return ( PVectorX<ReactiveSeq<T>>)PersistentCollectionX.super.permutations();
	}


	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#combinations(int)
	 */
	@Override
	default PVectorX<ReactiveSeq<T>> combinations(int size) {
		
		return (PVectorX<ReactiveSeq<T>>)PersistentCollectionX.super.combinations(size);
	}


	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#combinations()
	 */
	@Override
	default PVectorX<ReactiveSeq<T>> combinations() {
		
		return (PVectorX<ReactiveSeq<T>>)PersistentCollectionX.super.combinations();
	}

	default PVectorX<ListX<T>> sliding(int windowSize){
		return  (PVectorX<ListX<T>>)PersistentCollectionX.super.sliding(windowSize);
	}
	default PVectorX<ListX<T>> sliding(int windowSize, int increment){
		return  (PVectorX<ListX<T>>)PersistentCollectionX.super.sliding(windowSize,increment);
	}
	default PVectorX<T> scanLeft(Monoid<T> monoid){
		return  (PVectorX<T>)PersistentCollectionX.super.scanLeft(monoid);
	}
	default <U> PVectorX<U> scanLeft(U seed, BiFunction<? super U, ? super T, ? extends U> function){
		return  (PVectorX<U>)PersistentCollectionX.super.scanLeft(seed,function);
	}
	default PVectorX<T> scanRight(Monoid<T> monoid){
		return  (PVectorX<T>)PersistentCollectionX.super.scanRight(monoid);
	}
	default <U> PVectorX<U> scanRight(U identity, BiFunction<? super T, ? super U,? extends U> combiner){
		return  (PVectorX<U>)PersistentCollectionX.super.scanRight(identity,combiner);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#plusInOrder(java.lang.Object)
	 */
	@Override
	default PVectorX<T> plusInOrder(T e) {
		
		return (PVectorX<T>)PersistentCollectionX.super.plusInOrder(e);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#cycle(int)
	 */
	@Override
	default PVectorX<T> cycle(int times) {
		
		return (PVectorX<T>)PersistentCollectionX.super.cycle(times);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#cycle(com.aol.cyclops.sequence.Monoid, int)
	 */
	@Override
	default PVectorX<T> cycle(Monoid<T> m, int times) {
		
		return (PVectorX<T>)PersistentCollectionX.super.cycle(m, times);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#cycleWhile(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> cycleWhile(Predicate<? super T> predicate) {
		
		return (PVectorX<T>)PersistentCollectionX.super.cycleWhile(predicate);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#cycleUntil(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> cycleUntil(Predicate<? super T> predicate) {
		
		return (PVectorX<T>)PersistentCollectionX.super.cycleUntil(predicate);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zipStream(java.util.stream.Stream)
	 */
	@Override
	default <U> PVectorX<Tuple2<T, U>> zip(Stream<? extends U> other) {
		
		return (PVectorX)PersistentCollectionX.super.zip(other);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zip(org.jooq.lambda.Seq)
	 */
	@Override
	default <U> PVectorX<Tuple2<T, U>> zip(Seq<? extends U> other) {
		
		return (PVectorX)PersistentCollectionX.super.zip(other);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zip3(java.util.stream.Stream, java.util.stream.Stream)
	 */
	@Override
	default <S, U> PVectorX<Tuple3<T, S, U>> zip3(Stream<? extends S> second, Stream<? extends U> third) {
		
		return (PVectorX)PersistentCollectionX.super.zip3(second, third);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zip4(java.util.stream.Stream, java.util.stream.Stream, java.util.stream.Stream)
	 */
	@Override
	default <T2, T3, T4> PVectorX<Tuple4<T, T2, T3, T4>> zip4(Stream<? extends T2> second, Stream<? extends T3> third,
			Stream<? extends T4> fourth) {
		
		return (PVectorX)PersistentCollectionX.super.zip4(second, third, fourth);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zipWithIndex()
	 */
	@Override
	default PVectorX<Tuple2<T, Long>> zipWithIndex() {
		
		return (PVectorX<Tuple2<T, Long>>)PersistentCollectionX.super.zipWithIndex();
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#distinct()
	 */
	@Override
	default PVectorX<T> distinct() {
		
		return (PVectorX<T>)PersistentCollectionX.super.distinct();
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#sorted()
	 */
	@Override
	default PVectorX<T> sorted() {
		
		return (PVectorX<T>)PersistentCollectionX.super.sorted();
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#sorted(java.util.Comparator)
	 */
	@Override
	default PVectorX<T> sorted(Comparator<? super T> c) {
		
		return (PVectorX<T>)PersistentCollectionX.super.sorted(c);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#skipWhile(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> skipWhile(Predicate<? super T> p) {
		
		return (PVectorX<T>)PersistentCollectionX.super.skipWhile(p);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#skipUntil(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> skipUntil(Predicate<? super T> p) {
		
		return (PVectorX<T>)PersistentCollectionX.super.skipUntil(p);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#limitWhile(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> limitWhile(Predicate<? super T> p) {
		
		return (PVectorX<T>)PersistentCollectionX.super.limitWhile(p);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#limitUntil(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> limitUntil(Predicate<? super T> p) {
		
		return (PVectorX<T>)PersistentCollectionX.super.limitUntil(p);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#intersperse(java.lang.Object)
	 */
	@Override
	default PVectorX<T> intersperse(T value) {
		
		return (PVectorX<T>)PersistentCollectionX.super.intersperse(value);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#shuffle()
	 */
	@Override
	default PVectorX<T> shuffle() {
		
		return (PVectorX<T>)PersistentCollectionX.super.shuffle();
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#skipLast(int)
	 */
	@Override
	default PVectorX<T> skipLast(int num) {
		
		return (PVectorX<T>)PersistentCollectionX.super.skipLast(num);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#limitLast(int)
	 */
	@Override
	default PVectorX<T> limitLast(int num) {
		
		return (PVectorX<T>)PersistentCollectionX.super.limitLast(num);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#onEmpty(java.lang.Object)
	 */
	@Override
	default PVectorX<T> onEmpty(T value) {
		
		return (PVectorX<T>)PersistentCollectionX.super.onEmpty(value);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#onEmptyGet(java.util.function.Supplier)
	 */
	@Override
	default PVectorX<T> onEmptyGet(Supplier<? extends T> supplier) {
		
		return (PVectorX<T>)PersistentCollectionX.super.onEmptyGet(supplier);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#onEmptyThrow(java.util.function.Supplier)
	 */
	@Override
	default <X extends Throwable> PVectorX<T> onEmptyThrow(Supplier<? extends X> supplier) {
		
		return (PVectorX<T>)PersistentCollectionX.super.onEmptyThrow(supplier);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#shuffle(java.util.Random)
	 */
	@Override
	default PVectorX<T> shuffle(Random random) {
		
		return (PVectorX<T>)PersistentCollectionX.super.shuffle(random);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#ofType(java.lang.Class)
	 */
	@Override
	default <U> PVectorX<U> ofType(Class<? extends U> type) {
		
		return (PVectorX<U>)PersistentCollectionX.super.ofType(type);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#filterNot(java.util.function.Predicate)
	 */
	@Override
	default PVectorX<T> filterNot(Predicate<? super T> fn) {
		
		return (PVectorX<T>)PersistentCollectionX.super.filterNot(fn);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#notNull()
	 */
	@Override
	default PVectorX<T> notNull() {
		
		return (PVectorX<T>)PersistentCollectionX.super.notNull();
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#removeAll(java.util.stream.Stream)
	 */
	@Override
	default PVectorX<T> removeAll(Stream<? extends T> stream) {
		
		return (PVectorX<T>)PersistentCollectionX.super.removeAll(stream);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#removeAll(java.lang.Iterable)
	 */
	@Override
	default PVectorX<T> removeAll(Iterable<? extends T> it) {
		
		return (PVectorX<T>)PersistentCollectionX.super.removeAll(it);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#removeAll(java.lang.Object[])
	 */
	@Override
	default PVectorX<T> removeAll(T... values) {
		
		return (PVectorX<T>)PersistentCollectionX.super.removeAll(values);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#retainAll(java.lang.Iterable)
	 */
	@Override
	default PVectorX<T> retainAll(Iterable<? extends T> it) {
		
		return (PVectorX<T>)PersistentCollectionX.super.retainAll(it);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#retainAll(java.util.stream.Stream)
	 */
	@Override
	default PVectorX<T> retainAll(Stream<? extends T> seq) {
		
		return (PVectorX<T>)PersistentCollectionX.super.retainAll(seq);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#retainAll(java.lang.Object[])
	 */
	@Override
	default PVectorX<T> retainAll(T... values) {
		
		return (PVectorX<T>)PersistentCollectionX.super.retainAll(values);
	}
	
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#cast(java.lang.Class)
	 */
	@Override
	default <U> PVectorX<U> cast(Class<? extends U> type) {
		
		return (PVectorX<U>)PersistentCollectionX.super.cast(type);
	}
	
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#patternMatch(java.lang.Object, java.util.function.Function)
	 */
	@Override
	default <R> PVectorX<R> patternMatch(
			Function<CheckValue1<T, R>, CheckValue1<T, R>> case1,Supplier<? extends R> otherwise) {
		
		return (PVectorX<R>)PersistentCollectionX.super.patternMatch(case1,otherwise);
	}
	 @Override
	    default <C extends Collection<? super T>> PVectorX<C> grouped(int size, Supplier<C> supplier) {
	        
	        return (PVectorX<C>)PersistentCollectionX.super.grouped(size, supplier);
	    }


	    @Override
	    default PVectorX<ListX<T>> groupedUntil(Predicate<? super T> predicate) {
	        
	        return (PVectorX<ListX<T>>)PersistentCollectionX.super.groupedUntil(predicate);
	    }


	    @Override
	    default PVectorX<ListX<T>> groupedStatefullyWhile(BiPredicate<ListX<? super T>, ? super T> predicate) {
	        
	        return (PVectorX<ListX<T>>)PersistentCollectionX.super.groupedStatefullyWhile(predicate);
	    }


	    @Override
	    default PVectorX<ListX<T>> groupedWhile(Predicate<? super T> predicate) {
	        
	        return (PVectorX<ListX<T>>)PersistentCollectionX.super.groupedWhile(predicate);
	    }


	    @Override
	    default <C extends Collection<? super T>> PVectorX<C> groupedWhile(Predicate<? super T> predicate,
	            Supplier<C> factory) {
	        
	        return (PVectorX<C>)PersistentCollectionX.super.groupedWhile(predicate, factory);
	    }


	    @Override
	    default <C extends Collection<? super T>> PVectorX<C> groupedUntil(Predicate<? super T> predicate,
	            Supplier<C> factory) {
	        
	        return (PVectorX<C>)PersistentCollectionX.super.groupedUntil(predicate, factory);
	    }
	    @Override
	    default PVectorX<T> removeAll(Seq<? extends T> stream) {
	       
	        return (PVectorX<T>)PersistentCollectionX.super.removeAll(stream);
	    }


	    @Override
	    default PVectorX<T> retainAll(Seq<? extends T> stream) {
	       
	        return (PVectorX<T>)PersistentCollectionX.super.retainAll(stream);
	    }
	    
}
