package com.aol.cyclops.data.collections.extensions.persistent;

import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Optional;
import java.util.Random;
import java.util.function.BiFunction;
import java.util.function.BiPredicate;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.stream.Collector;
import java.util.stream.Stream;

import org.jooq.lambda.Seq;
import org.jooq.lambda.tuple.Tuple2;
import org.jooq.lambda.tuple.Tuple3;
import org.jooq.lambda.tuple.Tuple4;
import org.pcollections.HashTreePSet;
import org.pcollections.PSet;
import org.reactivestreams.Publisher;

import com.aol.cyclops.Monoid;
import com.aol.cyclops.Reducer;
import com.aol.cyclops.Reducers;
import com.aol.cyclops.control.Matchable.CheckValue1;
import com.aol.cyclops.control.ReactiveSeq;
import com.aol.cyclops.control.Trampoline;
import com.aol.cyclops.data.collections.extensions.standard.ListX;

public interface PSetX<T> extends PSet<T>, PersistentCollectionX<T>{
	
    /**
     * Create a PSetX that contains the Integers between start and end
     * 
     * @param start
     *            Number of range to start from
     * @param end
     *            Number for range to end at
     * @return Range PSetX
     */
    public static PSetX<Integer> range(int start, int end) {
        return ReactiveSeq.range(start,end).toPSetX();
    }
    /**
     * Create a PSetX that contains the Longs between start and end
     * 
     * @param start
     *            Number of range to start from
     * @param end
     *            Number for range to end at
     * @return Range PSetX
     */
    public static PSetX<Long> rangeLong(long start, long end) {
        return ReactiveSeq.rangeLong(start,end).toPSetX();
    }
    /**
     * Unfold a function into a PSetX
     * 
     * <pre>
     * {@code 
     *  PSetX.unfold(1,i->i<=6 ? Optional.of(Tuple.tuple(i,i+1)) : Optional.empty());
     * 
     * //(1,2,3,4,5) in any order
     * 
     * }</code>
     * 
     * @param seed Initial value 
     * @param unfolder Iteratively applied function, terminated by an empty Optional
     * @return PSetX generated by unfolder function
     */
    static <U,T> PSetX<T> unfold(U seed,Function<? super U,Optional<Tuple2<T,U>>> unfolder){
        return ReactiveSeq.unfold(seed, unfolder).toPSetX();
    }
    /**
     * Generate a PSetX from the provided Supplier up to the provided limit number of times
     * 
     * @param limit Max number of elements to generate
     * @param s Supplier to generate PSetX elements
     * @return PSetX generated from the provided Supplier
     */
    public static <T> PSetX<T> generate(long limit,Supplier<T> s){

        return ReactiveSeq.generate(s).limit(limit).toPSetX();
     }
    /**
     * Create a PSetX by iterative application of a function to an initial element up to the supplied limit number of times
     * 
     * @param limit Max number of elements to generate
     * @param seed Initial element
     * @param f Iteratively applied to each element to generate the next element
     * @return PSetX generated by iterative application
     */
    public static <T> PSetX<T> iterate(long limit, final T seed, final UnaryOperator<T> f) {
        return ReactiveSeq.iterate(seed, f).limit(limit).toPSetX();
        
    }
    
	public static <T> PSetX<T> of(T...values){
		
		return new PSetXImpl<>(HashTreePSet.from(Arrays.asList(values)));
	}
	public static <T> PSetX<T> empty(){
		return new PSetXImpl<>(HashTreePSet .empty());
	}
	public static <T> PSetX<T> singleton(T value){
		return new PSetXImpl<>(HashTreePSet.singleton(value));
	}
	public static<T> PSetX<T> fromIterable(Iterable<T> iterable){
		if(iterable instanceof PSetX)
			return (PSetX)iterable;
		if(iterable instanceof PSet)
			return new PSetXImpl<>((PSet)(iterable));
		PSet<T> res = HashTreePSet.<T>empty();
		Iterator<T> it = iterable.iterator();
		while(it.hasNext())
			res = res.plus(it.next());
		
		return new PSetXImpl<>(res);
	}
    /**
     * Construct a PSetX from an Publisher
     * 
     * @param publisher
     *            to construct PSetX from
     * @return PSetX
     */
    public static <T> PSetX<T> fromPublisher(Publisher<? extends T> publisher) {
        return ReactiveSeq.fromPublisher((Publisher<T>)publisher).toPSetX();
    }
	public static<T> PSetX<T> fromCollection(Collection<T> stream){
		if(stream instanceof PSetX)
			return (PSetX)(stream);
		if(stream instanceof PSet)
			return new PSetXImpl<>((PSet)(stream));
		return new PSetXImpl<>(HashTreePSet.from(stream));
	}
	public static<T> PSetX<T> fromStream(Stream<T> stream){
		return Reducers.<T>toPSetX().mapReduce(stream);
	}
	@Override
	default PSetX<T> toPSetX() {
		return this;
	}
	
	  /**
     * Combine two adjacent elements in a PSetX using the supplied BinaryOperator
     * This is a stateful grouping & reduction operation. The output of a combination may in turn be combined
     * with it's neighbor
     * <pre>
     * {@code 
     *  PSetX.of(1,1,2,3)
                   .combine((a, b)->a.equals(b),Semigroups.intSum)
                   .toListX()
                   
     *  //ListX(3,4) 
     * }</pre>
     * 
     * @param predicate Test to see if two neighbors should be joined
     * @param op Reducer to combine neighbors
     * @return Combined / Partially Reduced PSetX
     */
    default PSetX<T> combine(BiPredicate<? super T, ? super T> predicate, BinaryOperator<T> op){
        return (PSetX<T>)PersistentCollectionX.super.combine(predicate,op);
    }
	@Override
	default<R> PSetX<R> unit(Collection<R> col){
		return fromCollection(col);
	}
	@Override
	default <R> PSetX<R> unit(R value){
		return singleton(value);
	}
	@Override
	default <R> PSetX<R> unitIterator(Iterator<R> it){
		return fromIterable(()->it);
	}
	@Override
	default<R> PSetX<R> emptyUnit(){
		return empty();
	}
	@Override
	default ReactiveSeq<T> stream(){
		
		return ReactiveSeq.fromIterable(this);
	}
	default PSet<T> toPSet(){
		return this;
	}
	
	default <X> PSetX<X> from(Collection<X> col){
		return fromCollection(col);
	}
	default <T> Reducer<PSet<T>> monoid(){
		return Reducers.toPSet();
	}
	
	/* (non-Javadoc)
	 * @see org.pcollections.PSet#plus(java.lang.Object)
	 */
	@Override
	public PSetX<T> plus(T e);
	/* (non-Javadoc)
	 * @see org.pcollections.PSet#plusAll(java.util.Collection)
	 */
	@Override
	public PSetX<T> plusAll(Collection<? extends T> list) ;
	/* (non-Javadoc)
	 * @see org.pcollections.PSet#minus(java.lang.Object)
	 */
	@Override
	public PSetX<T> minus(Object e);
	/* (non-Javadoc)
	 * @see org.pcollections.PSet#minusAll(java.util.Collection)
	 */
	@Override
	public PSetX<T> minusAll(Collection<?> list);

	

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#reverse()
	 */
	@Override
	default PSetX<T> reverse() {
		return (PSetX<T>)PersistentCollectionX.super.reverse();
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#filter(java.util.function.Predicate)
	 */
	@Override
	default PSetX<T> filter(Predicate<? super T> pred) {
		return (PSetX<T>)PersistentCollectionX.super.filter(pred);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#map(java.util.function.Function)
	 */
	@Override
	default <R> PSetX<R> map(Function<? super T, ? extends R> mapper) {
		return (PSetX<R>)PersistentCollectionX.super.map(mapper);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#flatMap(java.util.function.Function)
	 */
	@Override
	default <R> PSetX<R> flatMap(Function<? super T, ? extends Iterable<? extends R>> mapper) {
		return (PSetX<R>)PersistentCollectionX.super.flatMap(mapper);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#limit(long)
	 */
	@Override
	default PSetX<T> limit(long num) {
		return (PSetX<T>)PersistentCollectionX.super.limit(num);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#skip(long)
	 */
	@Override
	default PSetX<T> skip(long num) {
		return (PSetX<T>)PersistentCollectionX.super.skip(num);
	}
	default PSetX<T> takeRight(int num){
		return (PSetX<T>)PersistentCollectionX.super.takeRight(num);
	}
	default PSetX<T> dropRight(int num){
		return (PSetX<T>)PersistentCollectionX.super.dropRight(num);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#takeWhile(java.util.function.Predicate)
	 */
	@Override
	default PSetX<T> takeWhile(Predicate<? super T> p) {
		return (PSetX<T>)PersistentCollectionX.super.takeWhile(p);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#dropWhile(java.util.function.Predicate)
	 */
	@Override
	default PSetX<T> dropWhile(Predicate<? super T> p) {
		return (PSetX<T>)PersistentCollectionX.super.dropWhile(p);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#takeUntil(java.util.function.Predicate)
	 */
	@Override
	default PSetX<T> takeUntil(Predicate<? super T> p) {
		return (PSetX<T>)PersistentCollectionX.super.takeUntil(p);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#dropUntil(java.util.function.Predicate)
	 */
	@Override
	default PSetX<T> dropUntil(Predicate<? super T> p) {
		return (PSetX<T>)PersistentCollectionX.super.dropUntil(p);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#trampoline(java.util.function.Function)
	 */
	@Override
	default <R> PSetX<R> trampoline(Function<? super T, ? extends Trampoline<? extends R>> mapper) {
		return (PSetX<R>)PersistentCollectionX.super.trampoline(mapper);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#slice(long, long)
	 */
	@Override
	default PSetX<T> slice(long from, long to) {
		return (PSetX<T>)PersistentCollectionX.super.slice(from, to);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#sorted(java.util.function.Function)
	 */
	@Override
	default <U extends Comparable<? super U>> PSetX<T> sorted(Function<? super T, ? extends U> function) {
		return (PSetX<T>)PersistentCollectionX.super.sorted(function);
	}
	default PSetX<ListX<T>> grouped(int groupSize){
		return  (PSetX<ListX<T>>)PersistentCollectionX.super.grouped(groupSize);
	}
	default <K, A, D> PSetX<Tuple2<K, D>> grouped(Function<? super T, ? extends K> classifier, Collector<? super T, A, D> downstream){
		return  (PSetX)PersistentCollectionX.super.grouped(classifier,downstream);
	}
	default <K> PSetX<Tuple2<K, Seq<T>>> grouped(Function<? super T, ? extends K> classifier){
		return  (PSetX)PersistentCollectionX.super.grouped(classifier);
	}
	default <U> PSetX<Tuple2<T, U>> zip(Iterable<? extends U> other){
		return  (PSetX)PersistentCollectionX.super.zip(other);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zip(java.lang.Iterable, java.util.function.BiFunction)
	 */
	@Override
	default <U, R> PSetX<R> zip(Iterable<? extends U> other,
			BiFunction<? super T, ? super U, ? extends R> zipper) {
		
		return (PSetX<R>)PersistentCollectionX.super.zip(other, zipper);
	}
	@Override
    default <U, R> PSetX<R> zip(Seq<? extends U> other,
            BiFunction<? super T, ? super U, ? extends R> zipper) {
        
        return (PSetX<R>)PersistentCollectionX.super.zip(other, zipper);
    }
	@Override
    default <U, R> PSetX<R> zip(Stream<? extends U> other,
            BiFunction<? super T, ? super U, ? extends R> zipper) {
        
        return (PSetX<R>)PersistentCollectionX.super.zip(other, zipper);
    }

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#permutations()
	 */
	@Override
	default PSetX<ReactiveSeq<T>> permutations() {
		
		return ( PSetX<ReactiveSeq<T>>)PersistentCollectionX.super.permutations();
	}


	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#combinations(int)
	 */
	@Override
	default PSetX<ReactiveSeq<T>> combinations(int size) {
		
		return (PSetX<ReactiveSeq<T>>)PersistentCollectionX.super.combinations(size);
	}


	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#combinations()
	 */
	@Override
	default PSetX<ReactiveSeq<T>> combinations() {
		
		return (PSetX<ReactiveSeq<T>>)PersistentCollectionX.super.combinations();
	}

	default PSetX<ListX<T>> sliding(int windowSize){
		return  (PSetX<ListX<T>>)PersistentCollectionX.super.sliding(windowSize);
	}
	default PSetX<ListX<T>> sliding(int windowSize, int increment){
		return  (PSetX<ListX<T>>)PersistentCollectionX.super.sliding(windowSize,increment);
	}
	default PSetX<T> scanLeft(Monoid<T> monoid){
		return  (PSetX<T>)PersistentCollectionX.super.scanLeft(monoid);
	}
	default <U> PSetX<U> scanLeft(U seed, BiFunction<? super U, ? super T, ? extends U> function){
		return  (PSetX<U>)PersistentCollectionX.super.scanLeft(seed,function);
	}
	default PSetX<T> scanRight(Monoid<T> monoid){
		return  (PSetX<T>)PersistentCollectionX.super.scanRight(monoid);
	}
	default <U> PSetX<U> scanRight(U identity, BiFunction<? super T, ? super U,? extends U> combiner){
		return  (PSetX<U>)PersistentCollectionX.super.scanRight(identity,combiner);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#plusInOrder(java.lang.Object)
	 */
	@Override
	default PSetX<T> plusInOrder(T e) {
		
		return (PSetX<T>)PersistentCollectionX.super.plusInOrder(e);
	}
	/* (non-Javadoc)
     * @see com.aol.cyclops.collections.extensions.standard.MutableCollectionX#cycle(int)
     */
    @Override
    default PStackX<T> cycle(int times) {
        
        return this.stream().cycle(times).toPStackX();
    }
    /* (non-Javadoc)
     * @see com.aol.cyclops.collections.extensions.standard.MutableCollectionX#cycle(com.aol.cyclops.sequence.Monoid, int)
     */
    @Override
    default PStackX<T> cycle(Monoid<T> m, int times) {
        
        return this.stream().cycle(m,times).toPStackX();
    }
    /* (non-Javadoc)
     * @see com.aol.cyclops.collections.extensions.standard.MutableCollectionX#cycleWhile(java.util.function.Predicate)
     */
    @Override
    default PStackX<T> cycleWhile(Predicate<? super T> predicate) {
        
        return this.stream().cycleWhile(predicate).toPStackX();
    }
    /* (non-Javadoc)
     * @see com.aol.cyclops.collections.extensions.standard.MutableCollectionX#cycleUntil(java.util.function.Predicate)
     */
    @Override
    default PStackX<T> cycleUntil(Predicate<? super T> predicate) {
        
        return this.stream().cycleUntil(predicate).toPStackX();
    }
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zip(java.util.stream.Stream)
	 */
	@Override
	default <U> PSetX<Tuple2<T, U>> zip(Stream<? extends U> other) {
		
		return (PSetX)PersistentCollectionX.super.zip(other);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zip(org.jooq.lambda.Seq)
	 */
	@Override
	default <U> PSetX<Tuple2<T, U>> zip(Seq<? extends U> other) {
		
		return (PSetX)PersistentCollectionX.super.zip(other);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zip3(java.util.stream.Stream, java.util.stream.Stream)
	 */
	@Override
	default <S, U> PSetX<Tuple3<T, S, U>> zip3(Stream<? extends S> second, Stream<? extends U> third) {
		
		return (PSetX)PersistentCollectionX.super.zip3(second, third);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zip4(java.util.stream.Stream, java.util.stream.Stream, java.util.stream.Stream)
	 */
	@Override
	default <T2, T3, T4> PSetX<Tuple4<T, T2, T3, T4>> zip4(Stream<? extends T2> second, Stream<? extends T3> third,
			Stream<? extends T4> fourth) {
		
		return (PSetX)PersistentCollectionX.super.zip4(second, third, fourth);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#zipWithIndex()
	 */
	@Override
	default PSetX<Tuple2<T, Long>> zipWithIndex() {
		
		return (PSetX<Tuple2<T, Long>>)PersistentCollectionX.super.zipWithIndex();
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#distinct()
	 */
	@Override
	default PSetX<T> distinct() {
		
		return (PSetX<T>)PersistentCollectionX.super.distinct();
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#sorted()
	 */
	@Override
	default PSetX<T> sorted() {
		
		return (PSetX<T>)PersistentCollectionX.super.sorted();
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#sorted(java.util.Comparator)
	 */
	@Override
	default PSetX<T> sorted(Comparator<? super T> c) {
		
		return (PSetX<T>)PersistentCollectionX.super.sorted(c);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#skipWhile(java.util.function.Predicate)
	 */
	@Override
	default PSetX<T> skipWhile(Predicate<? super T> p) {
		
		return (PSetX<T>)PersistentCollectionX.super.skipWhile(p);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#skipUntil(java.util.function.Predicate)
	 */
	@Override
	default PSetX<T> skipUntil(Predicate<? super T> p) {
		
		return (PSetX<T>)PersistentCollectionX.super.skipUntil(p);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#limitWhile(java.util.function.Predicate)
	 */
	@Override
	default PSetX<T> limitWhile(Predicate<? super T> p) {
		
		return (PSetX<T>)PersistentCollectionX.super.limitWhile(p);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#limitUntil(java.util.function.Predicate)
	 */
	@Override
	default PSetX<T> limitUntil(Predicate<? super T> p) {
		
		return (PSetX<T>)PersistentCollectionX.super.limitUntil(p);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#intersperse(java.lang.Object)
	 */
	@Override
	default PSetX<T> intersperse(T value) {
		
		return (PSetX<T>)PersistentCollectionX.super.intersperse(value);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#shuffle()
	 */
	@Override
	default PSetX<T> shuffle() {
		
		return (PSetX<T>)PersistentCollectionX.super.shuffle();
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#skipLast(int)
	 */
	@Override
	default PSetX<T> skipLast(int num) {
		
		return (PSetX<T>)PersistentCollectionX.super.skipLast(num);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#limitLast(int)
	 */
	@Override
	default PSetX<T> limitLast(int num) {
		
		return (PSetX<T>)PersistentCollectionX.super.limitLast(num);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#onEmpty(java.lang.Object)
	 */
	@Override
	default PSetX<T> onEmpty(T value) {
		
		return (PSetX<T>)PersistentCollectionX.super.onEmpty(value);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#onEmptyGet(java.util.function.Supplier)
	 */
	@Override
	default PSetX<T> onEmptyGet(Supplier<? extends T> supplier) {
		
		return (PSetX<T>)PersistentCollectionX.super.onEmptyGet(supplier);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#onEmptyThrow(java.util.function.Supplier)
	 */
	@Override
	default <X extends Throwable> PSetX<T> onEmptyThrow(Supplier<? extends X> supplier) {
		
		return (PSetX<T>)PersistentCollectionX.super.onEmptyThrow(supplier);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#shuffle(java.util.Random)
	 */
	@Override
	default PSetX<T> shuffle(Random random) {
		
		return (PSetX<T>)PersistentCollectionX.super.shuffle(random);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#ofType(java.lang.Class)
	 */
	@Override
	default <U> PSetX<U> ofType(Class<? extends U> type) {
		
		return (PSetX<U>)PersistentCollectionX.super.ofType(type);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#filterNot(java.util.function.Predicate)
	 */
	@Override
	default PSetX<T> filterNot(Predicate<? super T> fn) {
		
		return (PSetX<T>)PersistentCollectionX.super.filterNot(fn);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#notNull()
	 */
	@Override
	default PSetX<T> notNull() {
		
		return (PSetX<T>)PersistentCollectionX.super.notNull();
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#removeAll(java.util.stream.Stream)
	 */
	@Override
	default PSetX<T> removeAll(Stream<? extends T> stream) {
		
		return (PSetX<T>)PersistentCollectionX.super.removeAll(stream);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#removeAll(java.lang.Iterable)
	 */
	@Override
	default PSetX<T> removeAll(Iterable<? extends T> it) {
		
		return (PSetX<T>)PersistentCollectionX.super.removeAll(it);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#removeAll(java.lang.Object[])
	 */
	@Override
	default PSetX<T> removeAll(T... values) {
		
		return (PSetX<T>)PersistentCollectionX.super.removeAll(values);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#retainAll(java.lang.Iterable)
	 */
	@Override
	default PSetX<T> retainAll(Iterable<? extends T> it) {
		
		return (PSetX<T>)PersistentCollectionX.super.retainAll(it);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#retainAll(java.util.stream.Stream)
	 */
	@Override
	default PSetX<T> retainAll(Stream<? extends T> seq) {
		
		return (PSetX<T>)PersistentCollectionX.super.retainAll(seq);
	}
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#retainAll(java.lang.Object[])
	 */
	@Override
	default PSetX<T> retainAll(T... values) {
		
		return (PSetX<T>)PersistentCollectionX.super.retainAll(values);
	}

	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#cast(java.lang.Class)
	 */
	@Override
	default <U> PSetX<U> cast(Class<? extends U> type) {
		
		return (PSetX<U>)PersistentCollectionX.super.cast(type);
	}
	
	/* (non-Javadoc)
	 * @see com.aol.cyclops.collections.extensions.persistent.PersistentCollectionX#patternMatch(java.lang.Object, java.util.function.Function)
	 */
	@Override
	default <R> PSetX<R> patternMatch(
			Function<CheckValue1<T, R>, CheckValue1<T, R>> case1,Supplier<? extends R> otherwise) {
		
		return (PSetX<R>)PersistentCollectionX.super.patternMatch(case1,otherwise);
	}
	 @Override
	    default <C extends Collection<? super T>> PSetX<C> grouped(int size, Supplier<C> supplier) {
	        
	        return (PSetX<C>)PersistentCollectionX.super.grouped(size, supplier);
	    }


	    @Override
	    default PSetX<ListX<T>> groupedUntil(Predicate<? super T> predicate) {
	        
	        return (PSetX<ListX<T>>)PersistentCollectionX.super.groupedUntil(predicate);
	    }


	    @Override
	    default PSetX<ListX<T>> groupedStatefullyWhile(BiPredicate<ListX<? super T>, ? super T> predicate) {
	        
	        return (PSetX<ListX<T>>)PersistentCollectionX.super.groupedStatefullyWhile(predicate);
	    }


	    @Override
	    default PSetX<ListX<T>> groupedWhile(Predicate<? super T> predicate) {
	        
	        return (PSetX<ListX<T>>)PersistentCollectionX.super.groupedWhile(predicate);
	    }


	    @Override
	    default <C extends Collection<? super T>> PSetX<C> groupedWhile(Predicate<? super T> predicate,
	            Supplier<C> factory) {
	        
	        return (PSetX<C>)PersistentCollectionX.super.groupedWhile(predicate, factory);
	    }


	    @Override
	    default <C extends Collection<? super T>> PSetX<C> groupedUntil(Predicate<? super T> predicate,
	            Supplier<C> factory) {
	        
	        return (PSetX<C>)PersistentCollectionX.super.groupedUntil(predicate, factory);
	    }
	    @Override
	    default PSetX<T> removeAll(Seq<? extends T> stream) {
	       
	        return (PSetX<T>)PersistentCollectionX.super.removeAll(stream);
	    }


	    @Override
	    default PSetX<T> retainAll(Seq<? extends T> stream) {
	       
	        return (PSetX<T>)PersistentCollectionX.super.retainAll(stream);
	    }
	    
	
}
