/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package org.jetbrains.kotlin.analysis.api.components

import com.google.common.collect.Multimap
import com.intellij.psi.PsiElement
import org.jetbrains.kotlin.analysis.api.KtAnalysisNonPublicApi
import org.jetbrains.kotlin.analysis.api.lifetime.withValidityAssertion
import org.jetbrains.kotlin.psi.KtClassOrObject
import org.jetbrains.kotlin.psi.KtElement
import org.jetbrains.kotlin.psi.KtFile

@KtAnalysisNonPublicApi
public abstract class KtMetadataCalculator : KtAnalysisSessionComponent() {
    public abstract fun calculateMetadata(ktClass: KtClassOrObject, mapping: Multimap<KtElement, PsiElement>): Metadata

    public abstract fun calculateMetadata(ktFile: KtFile, mapping: Multimap<KtElement, PsiElement>): Metadata
}

@KtAnalysisNonPublicApi
public interface KtMetadataCalculatorMixIn : KtAnalysisSessionMixIn {
    /**
     * Calculates metadata that would be generated by the compiler in case this class was compiled to the JVM class file.
     *
     * @param mapping map containing the light elements ([KtLightElement]) for each callable declaration in this class.
     */
    public fun KtClassOrObject.calculateMetadata(mapping: Multimap<KtElement, PsiElement>): Metadata =
        withValidityAssertion { analysisSession.metadataCalculator.calculateMetadata(this, mapping) }

    /**
     * Calculates metadata that would be generated by the compiler in case this file was compiled to the JVM class file.
     *
     * @param mapping map containing the light elements ([KtLightElement]) for each callable declaration in this file.
     */
    public fun KtFile.calculateMetadata(mapping: Multimap<KtElement, PsiElement>): Metadata =
        withValidityAssertion { analysisSession.metadataCalculator.calculateMetadata(this, mapping) }
}
