/*
 * Copyright (C) 2015 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.support.test.rule.logging;

import android.app.Instrumentation;
import android.app.UiAutomation;
import android.content.Context;
import android.os.Build;
import android.os.ParcelFileDescriptor;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.support.test.InstrumentationRegistry;
import android.support.test.filters.SdkSuppress;
import android.text.TextUtils;
import android.util.Log;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import static android.support.test.internal.util.Checks.checkState;

/**
 * Convenience methods to ensure logging rules perform certain actions in the same manner.
 */
public class RuleLoggingUtils {

    private static final String TAG = "RuleLoggingUtils";

    public static final String LOGGING_SUB_DIR_NAME = "testdata";

    /**
     * Test utility method to check if a file contains the specified content.
     *
     * @param message       to be used when throwing an Assertion error if the content does not
     *                      match
     * @param file          to inspect
     * @param contentString to compare against the content of the {@code file}
     * @param contains      indicating whether the {@code contentString} should be found in file
     * @throws AssertionError when the content is or is not found depending on {@code contains}
     * @throws IOException    when the there are issues accessing the {@code file} parameter
     */
    private static void assertFileContent(@Nullable String message, File file,
            String contentString, boolean contains)
            throws AssertionError, IOException {
        StringBuilder fileContents = new StringBuilder();

        // These variables make the logic a little bit easier to understand.
        boolean shouldContain = contains;
        boolean didContain = false;
        try (
                FileReader fileReader = new FileReader(file);
                BufferedReader bufferedReader = new BufferedReader(fileReader)
        ) {
            String readLine;

            // Loop continues while data exists.
            while ((readLine = bufferedReader.readLine()) != null) {
                fileContents.append(readLine);
                fileContents.append(System.lineSeparator());

                if (fileContents.length() >= contentString.length()) {
                    long index = fileContents.indexOf(contentString);
                    if (!shouldContain && index >= 0) {
                        throw new AssertionError("File content found that shouldn't have been " +
                                "present, contentString=" + contentString);
                    } else if (contains && index >= 0) {
                        didContain = true;
                        // Found content so short-circuit out of content scanning.
                        break;
                    } else {
                        // Trim the fileContents to contentString as everything longer than that
                        // was already tested for a match.
                        fileContents.delete(0,
                                fileContents.length() - contentString.length());
                    }
                }
            }

            if (shouldContain && !didContain) {
                // If we didn't find the content we should throw an exception.
                throw new AssertionError("File content not found that should have been " +
                        "present, contentString=" + contentString);
            }
        } catch (AssertionError exception) {
            if (message != null) {
                throw new AssertionError(message, exception);
            } else {
                throw exception;
            }
        }
    }

    private static String getCommandFromParts(String[] commandParts) {
        StringBuilder commandBuilder = new StringBuilder();
        for (String commandPart : commandParts) {
            commandBuilder.append(commandPart);
            commandBuilder.append(" ");
        }
        return commandBuilder.toString();
    }

    private static File getTestDirectory(String className, String testName,
            @Nullable Integer testRunNumber) {
        Context context = InstrumentationRegistry.getTargetContext();
        File rootDir = context.getExternalFilesDir(null);

        // Create a test data subdirectory.
        File testFileDir = new File(rootDir, LOGGING_SUB_DIR_NAME);
        if (getTranslatedTestName(className, testName) != null) {
            testFileDir = new File(testFileDir, getTranslatedTestName(className, testName));
            // Append test run number if specified.
            if (testRunNumber != null) {
                testFileDir = new File(testFileDir, testRunNumber + "");
            }
        }

        if (!testFileDir.exists()) {
            if (!testFileDir.mkdirs()) {
                throw new RuntimeException("Unable to create logging rules log directory.");
            }
        }
        return testFileDir;
    }

    private static String getTranslatedTestName(String className, String testName) {
        if (className == null || testName == null) {
            return null;
        }
        String base = className + "_" + testName;

        // Shorten the common strings so "com.google.android" comes out as "c.g.a" for brevity.
        base = base.replace("com", "c")
                   .replace("google", "g")
                   .replace("android", "a")
                   .replace("perfmatters", "pm")
                   .replace("automatingperformancetesting", "apt");
        return base;
    }

    private static void writeProcessOutputToLogcat(Process process, String logTag)
            throws IOException {
        try (
                InputStreamReader inputStreamReader =
                        new InputStreamReader(process.getInputStream());
                BufferedReader bufferedReader = new BufferedReader(inputStreamReader)
        ) {
            String line;
            while (null != (line = bufferedReader.readLine())) {
                Log.w(logTag, line);
            }
        }
    }

    /**
     * Test utility method to check if a file is empty.
     *
     * @param message to be used when throwing an Assertion error if the content is not empty
     * @param file    to inspect
     * @throws AssertionError is thrown when the file isn't empty
     * @throws IOException    when the there are issues accessing the {@code file} parameter
     */
    public static void assertEmptyFile(@Nullable String message, File file)
            throws AssertionError, IOException {
        try {
            if (file.exists()) {
                try (
                        FileInputStream fis = new FileInputStream(file);
                        InputStreamReader isr = new InputStreamReader(fis);
                        BufferedReader br = new BufferedReader(isr)
                ) {
                    String firstLine = br.readLine();
                    if (firstLine != null && !TextUtils.isEmpty(firstLine)) {
                        throw new AssertionError(
                                "Expected file to be empty, but was able to read data: " +
                                        firstLine);
                    }
                }
            } else {
                throw new IOException("Expected file did not exist: " + file.getAbsolutePath());
            }
        } catch (IOException | AssertionError exception) {
            if (message != null) {
                throw new AssertionError(message, exception);
            } else {
                throw exception;
            }
        }
    }

    /**
     * Test utility method to check if a file contains the specified content.
     *
     * @param message       to be used when throwing an Assertion error if the content does not
     *                      match
     * @param file          to inspect
     * @param contentString to compare against the content of the {@code file}
     * @throws AssertionError is thrown when the content is not found
     * @throws IOException    when the there are issues accessing the {@code file} parameter
     */
    public static void assertFileContentContains(@Nullable String message, File file,
            String contentString) throws AssertionError, IOException {
        assertFileContent(message, file, contentString, true);
    }

    /**
     * Test utility method to check if a file doesn't contain the specified content.
     *
     * @param message       to be used when throwing an Assertion error if the content does not
     *                      match
     * @param file          to inspect
     * @param contentString to compare against the content of the {@code file}
     * @throws AssertionError is thrown when the content is not found
     * @throws IOException    when the there are issues accessing the {@code file} parameter
     */
    public static void assertFileContentDoesNotContain(@Nullable String message, File file,
            String contentString) throws AssertionError, IOException {
        assertFileContent(message, file, contentString, false);
    }

    /**
     * Test utility method to quickly check if a file begins with the specified content.
     *
     * @param message       to be used if the content does not match
     * @param file          to inspect
     * @param contentString to compare against the content of the {@code file}
     * @throws AssertionError is thrown when the content is not found
     * @throws IOException    when the there are issues accessing the {@code file} parameter
     */
    public static void assertFileContentStartsWith(@Nullable String message, File file,
            String contentString)
            throws AssertionError, IOException {
        StringBuilder fileContents = new StringBuilder();
        try (
                FileReader fileReader = new FileReader(file);
                BufferedReader bufferedReader = new BufferedReader(fileReader)
        ) {
            String readLine;
            // Loop continues while data exists and read data is shorter than the data to compare.
            while (((readLine = bufferedReader.readLine()) != null) &&
                    (fileContents.length() < contentString.length())) {
                fileContents.append(readLine);
                fileContents.append(System.lineSeparator());
            }
        }

        try {
            if (fileContents.length() < contentString.length()) {
                // Short-circuit if read content isn't long enough.
                throw new AssertionError("File content wasn't long enough to match, expected=" +
                        contentString + ", minimalStartingFileContent=" + fileContents);
            } else if (!fileContents.substring(0, contentString.length()).equals(contentString)) {
                throw new AssertionError("File content did not match, expected=" +
                        contentString + ", minimalStartingFileContent=" + fileContents);
            }
        } catch (AssertionError exception) {
            if (message != null) {
                throw new AssertionError(message, exception);
            } else {
                throw exception;
            }
        }
    }

    /**
     * Retrieve the directory where logging rules logs should be written to.
     * This directory is on external storage so it is not removed when the app is uninstalled. This
     * allows the files to be retrieved despite fatal (think OutOfMemory) exceptions.
     * {@code testRunNumber} should be set whenever a test method is run more than one time in a
     * single test run to indicate which iteration the logging is for. Use zero as a default.
     */
    public static File getTestDir(String className, String testName,
            int testRunNumber) {
        checkState(testRunNumber >= 0, "Invalid test run number (" + testRunNumber + ")");
        return getTestDirectory(className, testName, testRunNumber);
    }

    /**
     * Retrieve a file handle that is within the testing directory where tests should be written to.
     */
    public static File getTestFile(String className, String testName, String filename,
            int testRunNumber) {
        return new File(getTestDir(className, testName, testRunNumber), filename);
    }

    /**
     * Retrieve the test run directory where tests should be written to.
     */
    public static File getTestRunDir() {
        return getTestDirectory(null, null, null);
    }

    /**
     * Retrieve a file handle within the testing directory where test data can be written for the
     * complete test run.
     */
    public static File getTestRunFile(String filename) {
        return new File(getTestDirectory(null, null, null), filename);
    }

    /**
     * Utility method to print file to logcat for debugging purposes.
     */
    public static void printFileToLogcat(File logFile, String logcatTag) throws IOException {
        try (
                FileReader fileReader = new FileReader(logFile);
                BufferedReader bufferedReader = new BufferedReader(fileReader)
        ) {
            Log.w(logcatTag, "Logging file located at " + logFile.getAbsolutePath());
            String line;
            while (null != (line = bufferedReader.readLine())) {
                Log.w(logcatTag, line);
            }
        }
    }

    /**
     * Start a {@link Process} on the system using a process compatible with all Android runtimes.
     * Standard and error output is redirected to the specified file.
     * <p>
     * This command runs within the testing instrumentation and has some development permissions
     * already granted.
     *
     * @param commandParts the command and parameters to execute on the system
     * @param logFile      where comamnd output is written, or in the case of an error, the
     *                     exception output is written
     */
    @SdkSuppress(minSdkVersion = Build.VERSION_CODES.LOLLIPOP)
    public static void startCmdAndLogOutputPostL(String[] commandParts, File logFile) {
        try {
            Instrumentation testingInstrumentation = InstrumentationRegistry.getInstrumentation();
            UiAutomation uiAutomation = testingInstrumentation.getUiAutomation();
            String command = getCommandFromParts(commandParts);
            try (
                    InputStream inputStream = new ParcelFileDescriptor.AutoCloseInputStream(
                            uiAutomation.executeShellCommand(command));
                    InputStreamReader inputStreamReader = new InputStreamReader(inputStream);
                    BufferedReader reader = new BufferedReader(inputStreamReader);
                    FileWriter fileWriter = new FileWriter(logFile);
                    BufferedWriter bufferedWriter = new BufferedWriter(fileWriter)
            ) {
                String line;
                while (null != (line = reader.readLine())) {
                    bufferedWriter.write(line);
                    bufferedWriter.write(System.lineSeparator());
                }
            }
        } catch (Exception exception) {
            writeErrorToFileAndLogcat(logFile, TAG, "Couldn't start and write process output",
                    exception);
        }
    }

    /**
     * Start a {@link Process} with the command and arguments specified in {@code commandParts}.
     * <p>
     * You must call {@code Process.destroy()} on the object returned.
     */
    public static Process startProcess(String[] commandParts) throws IOException {
        ProcessBuilder processBuilder = new ProcessBuilder();
        processBuilder.command(commandParts);
        processBuilder.redirectErrorStream();
        return processBuilder.start();
    }

    /**
     * Start a {@link Process} on the system using either
     * {@link #startCmdAndLogOutputPostL(String[], File)} or
     * {@link #startProcessAndWriteOutputToFilePreL(String[], File)} according to the Android
     * version number passed in.
     * <p/>
     * This utility method eliminates the need to grant your app some system permissions when
     * running on Android Lollipop or above by using the test instrumentation to run the specified
     * command. If you are testing on pre-Lollipop devices you will need to ensure your test APK
     * has been granted any permissions needed to execute the commands passed in.
     *
     * @param commandParts   the command and parameters to execute on the system
     * @param logFile        where comamnd output is written, or in the case of an error, where the
     *                       exception output is written
     * @param androidVersion overrides the system Android version which is used to decide the best
     *                       method to invoke the command with. This is useful for code that needs
     *                       to write Android version specific tests.
     */
    public static void startProcessAndLogToFile(String[] commandParts, File logFile,
            @VisibleForTesting int androidVersion) {
        if (androidVersion > Build.VERSION_CODES.LOLLIPOP) {
            startCmdAndLogOutputPostL(commandParts, logFile);
        } else {
            startProcessAndWriteOutputToFilePreL(commandParts, logFile);
        }
    }

    /**
     * Start a {@link Process} on the system using a process compatible with all Android runtimes.
     * Standard and error output is redirected to the specified file.
     * <p/>
     * This command runs as the current user and requires appropriate permissions be granted to the
     * App/Test APK. If the runtime is Android M or greater use
     * {@link #startCmdAndLogOutputPostL} instead to run as with instrumentation
     * permissions.
     *
     * @param commandParts the command and parameters to execute on the system
     * @param logFile      where comamnd output is written, or in the case of an error, the
     *                     exception output is written
     */
    public static void startProcessAndWriteOutputToFilePreL(String[] commandParts, File logFile) {
        Process process = null;
        try {
            process = startProcess(commandParts);
            process.waitFor();
            writeProcessOutputToFile(process, logFile);
        } catch (InterruptedException | IOException exception) {
            writeErrorToFileAndLogcat(logFile, TAG, "Couldn't start and write process output",
                    exception);
        } finally {
            if (process != null) {
                process.destroy();
            }
        }
    }

    /**
     * Utility method to write an error message to a file and logcat as an error.
     */
    public static void writeErrorToFileAndLogcat(File file, String logTag, String errorMessage,
            @Nullable Exception exception) {
        if (exception != null) {
            Log.e(logTag, errorMessage, exception);
        } else {
            Log.e(logTag, errorMessage);
        }
        try (
                FileWriter fileWriter = new FileWriter(file)
        ) {
            fileWriter.append(errorMessage);
            fileWriter.append(System.lineSeparator());
            if (exception != null) {
                fileWriter.append(exception.toString());
            }
        } catch (IOException ioexception) {
            Log.e(logTag, "Unable to log error to file " + file.getAbsolutePath(), ioexception);
        }
    }

    /**
     * Utility method to read a {@link Process}'s output and write it to a file.
     */
    public static void writeProcessOutputToFile(Process process, File logFile) throws IOException {
        try (
                FileWriter fileWriter = new FileWriter(logFile);
                BufferedWriter bufferedWriter = new BufferedWriter(fileWriter);
                InputStreamReader inputStreamReader =
                        new InputStreamReader(process.getInputStream());
                BufferedReader bufferedReader = new BufferedReader(inputStreamReader)
        ) {
            String line;
            while (null != (line = bufferedReader.readLine())) {
                bufferedWriter.append(line);
                bufferedWriter.append(System.lineSeparator());
            }
        }
    }
}
