/*
 * Copyright (C) 2015 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.support.test.rule.logging;

import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.test.annotation.Beta;
import android.util.Log;

import org.junit.Rule;

import java.io.File;

import static android.support.test.rule.logging.RuleLoggingUtils.startProcess;
import static android.support.test.rule.logging.RuleLoggingUtils.startProcessAndLogToFile;

/**
 * When applied to a test class this {@code Rule} resets battery information ("stats") before
 * test method execution. The {@link Rule} then collects battery information after test
 * method execution and logs it to a file on external storage. The default file location is
 * maintained at {@link RuleLoggingUtils#getTestDir(String, String, int)}. The Rule performs
 * these actions through the execution of a dumpsys on the {@code batterystats} service.
 * <p/>
 * If the API level is less than 21 (Lollipop) this rule will do nothing since this dumpsys command
 * isn't supported. This TestRule has limited use for short running tests and is meant for very long
 * running tests.
 * <p/>
 * The {@link Rule}'s {@link #before()} and {@link #after()} methods can also be used for logging
 * battery information across a test suite by manually invoking the methods in a
 * {@link org.junit.runner.notification.RunListener RunListener}.
 * <p/>
 * Usage:
 * <code>
 * \@Rule
 * public LogBatteryInformationRule mLogBatteryInformationRule = new LogBatteryInformationRule();
 * <code>
 */
@Beta
public class LogBatteryInformationRule extends LoggingBaseRule {

    private static final String TAG = "LogBatteryInformationRule";

    private static final String DEFAULT_LOG_FILE_NAME = "battery.dumpsys.log";

    /**
     * {@inheritDoc}
     */
    public LogBatteryInformationRule() {

    }

    /**
     * {@inheritDoc}
     **/
    public LogBatteryInformationRule(@NonNull File logFileOutputDirectory,
            @Nullable String fileName) {
        super(logFileOutputDirectory, fileName);
    }

    @Override
    public void afterTest() {
        if (getAndroidRuntimeVersion() >= Build.VERSION_CODES.LOLLIPOP) {
            try {
                startProcessAndLogToFile(new String[] { "dumpsys", "batterystats"},
                        getLogFile(), getAndroidRuntimeVersion());
            } catch (Exception exception) {
                Log.e(TAG, "Unable to reset dumpsys", exception);
            }
        } else {
            Log.w(TAG, "Batterystats dumpsys not supported on API versions less than 21");
        }
    }

    @Override
    public void beforeTest() {
        if (getAndroidRuntimeVersion() >= Build.VERSION_CODES.LOLLIPOP) {
            Process process = null;
            try {
                process = startProcess(
                        new String[]{"dumpsys", "batterystats", "--reset"});
                process.waitFor();
            } catch (Exception exception) {
                Log.e(TAG, "Unable to reset dumpsys", exception);
            } finally {
                if (process != null) {
                    process.destroy();
                }
            }
        } else {
            Log.i(TAG, "Batterystats dumpsys not supported on API versions less than 21");
        }
    }

    public String getDefaultLogFileName() {
        return DEFAULT_LOG_FILE_NAME;
    }
}