package com.android.component.mvp.fragment.container;

import android.content.Context;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentActivity;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentTransaction;
import android.util.Log;
import android.view.View;

import com.android.component.mvp.fragment.IComponent;
import com.android.component.mvp.fragment.IComponentBehavior;
import com.android.component.mvp.fragment.MTComponent;
import com.android.component.mvp.fragment.config.BaseContainerConfig;
import com.android.component.mvp.fragment.config.ConfigMaker;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;

/**
 * @data         2018/7/24
 * @author       zhuqian(zaki)
 * @Description  component 抽象容器
 */
public abstract class AbstractComponentContainer<T extends BaseContainerConfig> extends MTComponent implements IComponentContainer {
    private static final String TAG = "ComponentContainer";
    private Map<Class<? extends IComponent>, IComponent> mChilds = new HashMap<>();
    private Queue<IComponent> mViewUnInitedChilds = new LinkedList<>(); //没有初始化View的Component
    private Class<? extends BaseContainerConfig> mConfigClazz;
    private T mConfig;
    private ConfigMaker mConfigMaker;

    public void setup(ConfigMaker configMaker) {
        this.mConfigMaker = configMaker;
        this.mConfigClazz = configMaker.ofConfig(getClass());
        try {
            mConfig = (T) this.mConfigClazz.newInstance();
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (java.lang.InstantiationException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        onContainerInitSuccess(getActivity(), getArguments());
        initComponentConfig();
    }

    private void onContainerInitSuccess(FragmentActivity activity, Bundle arguments) {
        if (mConfig != null) {
            mConfig.onContainerInit(activity, arguments);
        }
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        prepareCommitComponents();
    }

    private void prepareCommitComponents() {
        FragmentManager fragmentManager = getChildFragmentManager();
        FragmentTransaction fragmentTransaction = fragmentManager.beginTransaction();

        for (Map.Entry<Class<? extends IComponent>, IComponent> entry : mChilds.entrySet()) {
            Class<? extends IComponent> componentClz = entry.getKey();
            IComponent component = entry.getValue();
            int resId = mConfigMaker.wrapper(componentClz).resId(mConfigClazz);
            boolean initHidden = mConfigMaker.wrapper(componentClz).initHidden(mConfigClazz);
            Fragment fragment = component.getContent();
            Log.i(TAG, "commits content: "+ fragment + ", resId: " + resId + ", initHidden: " + initHidden);
            if (initHidden) {
                fragmentTransaction.add(resId, fragment, component.getClass().getSimpleName()).hide(fragment);
            } else {
                fragmentTransaction.replace(resId, fragment, component.getClass().getSimpleName());
            }
            mViewUnInitedChilds.add(component);
        }
        fragmentTransaction.commitAllowingStateLoss();
    }

    private void initComponentConfig() {
        if (mConfigClazz == null) {
            return;
        }
        List<Class<? extends IComponent>> components = mConfigMaker.ofComponents(mConfigClazz);

        for (Class<? extends IComponent> com : components) {
            IComponent component = mConfigMaker.ofComponent(com);
            component.attachContainer(this);
            mChilds.put(com, component);
        }
    }

    @Override
    public <C extends IComponent> C getComponent(Class<C> pClass) {
        IComponent component = mChilds.get(pClass);

        if (component == null) { //继续在Container里面找
            Collection<IComponent> components = mChilds.values();
            for (IComponent item : components) {
                if (item instanceof IComponentContainer) {
                    IComponent ret = ((IComponentContainer) item).getComponent(pClass);
                        if (ret != null) {
                            return (C) ret;
                        }
                }
            }
        }
        //
        return (C) component;
    }

    @Override
    public <B extends IComponentBehavior> B findBehavior(Class<B> bClass) {
        Collection<IComponent> components = mChilds.values();

        for (IComponent behavior : components) {
            Class<?>[] classes = behavior.getClass().getInterfaces();
            for (Class clz : classes) {
                if (clz == bClass) {
                    return (B) mChilds.get(behavior.getClass());
                }
            }
        }
        return null;
    }

    @Override
    public void onComponentViewCreated(Class<? extends IComponent> clz) {
        IComponent component = mChilds.get(clz);
        if (component != null && mViewUnInitedChilds.contains(component)) {
            Log.i(TAG, "onComponentViewCreated: " + clz.getSimpleName());
            mViewUnInitedChilds.remove(component); //view初始化成功就移除
        }

        if (mViewUnInitedChilds.isEmpty()) {
            //所有组件View初始化成功
            onContainerViewInitSuccess();
        }
    }

    /**
     * container 所有components初始化成功
     */
    protected void onContainerViewInitSuccess() {
        for (Map.Entry<Class<? extends IComponent>, IComponent> entry : mChilds.entrySet()) {
            entry.getValue().onContainerInflated();
        }
    }

    @Override
    public <C extends IComponent> boolean addComponent(Class<C> comClass) {
        IComponent component = mConfigMaker.ofComponent(comClass);
        if (component != null) {
            component.attachContainer(this);
            mChilds.put(comClass, component);

            FragmentManager fragmentManager = getChildFragmentManager();
            FragmentTransaction fragmentTransaction = fragmentManager.beginTransaction();


            int resId = mConfigMaker.wrapper(comClass).resId(mConfigClazz);
            boolean initHidden = mConfigMaker.wrapper(comClass).initHidden(mConfigClazz);
            Fragment fragment = component.getContent();
            Log.i(TAG, "addComponent content: "+ fragment + ", resId: " + resId + ", initHidden: " + initHidden);
            if (initHidden) {
                fragmentTransaction.detach(fragment).add(resId, fragment, component.getClass().getSimpleName());
            } else {
                fragmentTransaction.replace(resId, fragment, component.getClass().getSimpleName());
            }
            fragmentTransaction.commitAllowingStateLoss();
            return true;
        }
        return false;
    }


    @Override
    public <C extends IComponent> C removeComponent(Class<C> pClass) {
        IComponent component = mChilds.get(pClass);
        if (component != null) {
            Log.i(TAG, "removeComponent content: "+ component.getContent());
            getChildFragmentManager().beginTransaction().remove(component.getContent()).commitAllowingStateLoss();
        }
        return (C) component;
    }
}
