package com.amlzq.android.net;

import android.accounts.NetworkErrorException;
import android.os.Handler;
import android.os.Looper;

import com.amlzq.android.util.Logger;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by amlzq on 2017/7/14.
 * Android Native API
 * Http请求工具
 *
 * @author https://juejin.im/entry/57c4cf3befa631005ac93312
 */

public class HttpHelper {

    public static void post(String url, String content, ResultCallback callback) {
        HttpURLConnection connection = null;
        try {
            // 创建一个URL对象
            URL parsedUrl = new URL(url);

            // 调用URL的openConnection()方法,获取HttpURLConnection对象
            connection = (HttpURLConnection) parsedUrl.openConnection();

            connection.setRequestMethod("POST");// 设置请求方法为post
            connection.setConnectTimeout(10 * 1000);// 设置连接网络超时为10秒
            connection.setReadTimeout(5 * 1000);// 设置读取超时为5秒
            connection.setDoOutput(true);// 设置此方法,允许向服务器输出内容

            // post请求的参数
            String data = content;
            // 获得一个输出流,向服务器写数据,默认情况下,系统不允许向服务器输出内容
            OutputStream out = connection.getOutputStream();// 获得一个输出流,向服务器写数据
            out.write(data.getBytes());
            out.flush();
            out.close();

            int responseCode = connection.getResponseCode();// 调用此方法就不必再使用connection.connect()方法
            if (responseCode == HttpURLConnection.HTTP_OK) {
                InputStream is = connection.getInputStream();
                String response = getStringFromInputStream(is);
                successCallback(url, response, callback);
            } else {
                throw new NetworkErrorException("response status is " + responseCode);
            }
        } catch (Exception e) {
            e.printStackTrace();
            failureCallback(url, e, callback);
        } finally {
            if (connection != null) {
                connection.disconnect();// 关闭连接
            }
        }
    }

    public static void get(String url, ResultCallback callback) {
        HttpURLConnection connection = null;
        try {
            // 利用string url构建URL对象
            URL parsedUrl = new URL(url);
            connection = (HttpURLConnection) parsedUrl.openConnection();

            connection.setRequestMethod("GET");
            connection.setConnectTimeout(10 * 1000);
            connection.setReadTimeout(5 * 1000);

            int responseCode = connection.getResponseCode();
            if (responseCode == HttpURLConnection.HTTP_OK) {
                String response = getStringFromInputStream(connection.getInputStream());
                successCallback(url, response, callback);
            } else {
                throw new NetworkErrorException("response status is " + responseCode);
            }
        } catch (Exception e) {
            e.printStackTrace();
            failureCallback(url, e, callback);
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }

    public static void get(String url, Map<String, String> params, ResultCallback callback) {
        HttpURLConnection connection = null;
        try {
            StringBuilder tempParams = new StringBuilder();
            int pos = 0;
            for (String key : params.keySet()) {
                if (pos > 0) {
                    tempParams.append("&");
                }
                tempParams.append(String.format("%s=%s", key, URLEncoder.encode(params.get(key), "UTF-8")));
                pos++;
            }
            String requestUrl = url + "?" + tempParams.toString();
            Logger.d("request url: " + requestUrl);
            // 新建一个URL对象
            URL parsedUrl = new URL(requestUrl);
            // 打开一个HttpURLConnection连接
            connection = (HttpURLConnection) parsedUrl.openConnection();
            // 设置连接主机超时时间
            connection.setConnectTimeout(10 * 1000);
            //设置从主机读取数据超时
            connection.setReadTimeout(5 * 1000);
            // 设置是否使用缓存  默认是true
            connection.setUseCaches(true);
            // 设置为请求方法
            connection.setRequestMethod("GET");
            //connection设置请求头信息
            //设置请求中的媒体类型信息。
            connection.setRequestProperty("Content-Type", "application/json");
            //设置客户端与服务连接类型
            connection.addRequestProperty("Connection", "Keep-Alive");
            // 开始连接
            connection.connect();
            // 判断请求是否成功
            int responseCode = connection.getResponseCode();
            if (responseCode == HttpURLConnection.HTTP_OK) {
                // 获取返回的数据
                String response = streamToString(connection.getInputStream());
                successCallback(url, response, callback);
            } else {
                throw new NetworkErrorException("response status is " + responseCode);
            }
        } catch (Exception e) {
            e.printStackTrace();
            failureCallback(url, e, callback);
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }

    private static String getStringFromInputStream(InputStream is)
            throws IOException {
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        // 模板代码 必须熟练
        byte[] buffer = new byte[1024];
        int len = -1;
        while ((len = is.read(buffer)) != -1) {
            os.write(buffer, 0, len);
        }
        is.close();
        // 把流中的数据转换成字符串,采用的编码是UTF-8(模拟器默认编码)
        String state = os.toString();
        os.close();
        return state;
    }

    /**
     * @param is 从网络获取的输入流
     * @return 将输入流转换成字符串
     */
    public static String streamToString(InputStream is) {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            byte[] buffer = new byte[1024];
            int len = 0;
            while ((len = is.read(buffer)) != -1) {
                baos.write(buffer, 0, len);
            }
            baos.close();
            is.close();
            byte[] byteArray = baos.toByteArray();
            return new String(byteArray);
        } catch (Exception e) {
            Logger.e(e.getMessage(), e);
            return null;
        }
    }

    public static void downloadFile(String url, String filePath, ResultCallback callback) {
        try {
            // 新建一个URL对象
            URL parsedUrl = new URL(url);
            // 打开一个HttpURLConnection连接
            HttpURLConnection connection = (HttpURLConnection) parsedUrl.openConnection();
            // 设置连接主机超时时间
            connection.setConnectTimeout(5 * 1000);
            //设置从主机读取数据超时
            connection.setReadTimeout(5 * 1000);
            // 设置是否使用缓存  默认是true
            connection.setUseCaches(true);
            // 设置为Post请求
            connection.setRequestMethod("GET");
            //connection设置请求头信息
            //设置请求中的媒体类型信息。
            connection.setRequestProperty("Content-Type", "application/json");
            //设置客户端与服务连接类型
            connection.addRequestProperty("Connection", "Keep-Alive");
            // 开始连接
            connection.connect();
            // 判断请求是否成功
            if (connection.getResponseCode() == 200) {
                File descFile = new File(filePath);
                FileOutputStream fos = new FileOutputStream(descFile);
                byte[] buffer = new byte[1024];
                int len;
                InputStream inputStream = connection.getInputStream();
                while ((len = inputStream.read(buffer)) != -1) {
                    // 写到本地
                    fos.write(buffer, 0, len);
                }
            } else {
                Logger.e("download failure");
            }
            // 关闭连接
            connection.disconnect();
        } catch (Exception e) {
            Logger.e(e.getMessage(), e);
        }
    }

    public static void uploadFile(String url, String filePath, HashMap paramsMap, ResultCallback callback) {
        try {
            File file = new File(filePath);
            //新建url对象
            URL parsedUrl = new URL(url);
            //通过HttpURLConnection对象,向网络地址发送请求
            HttpURLConnection connection = (HttpURLConnection) parsedUrl.openConnection();
            //设置该连接允许读取
            connection.setDoOutput(true);
            //设置该连接允许写入
            connection.setDoInput(true);
            //设置不能适用缓存
            connection.setUseCaches(false);
            //设置连接超时时间
            connection.setConnectTimeout(5 * 1000);   //设置连接超时时间
            //设置读取超时时间
            connection.setReadTimeout(5 * 1000);   //读取超时
            //设置连接方法post
            connection.setRequestMethod("POST");
            //设置维持长连接
            connection.setRequestProperty("connection", "Keep-Alive");
            //设置文件字符集
            connection.setRequestProperty("Accept-Charset", "UTF-8");
            //设置文件类型
            connection.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + "*****");
            String name = file.getName();
            DataOutputStream requestStream = new DataOutputStream(connection.getOutputStream());
            requestStream.writeBytes("--" + "*****" + "\r\n");
            //发送文件参数信息
            StringBuilder tempParams = new StringBuilder();
            tempParams.append("Content-Disposition: form-data; name=\"" + name + "\"; filename=\"" + name + "\"; ");
            int pos = 0;
            int size = paramsMap.size();
            for (Object key : paramsMap.keySet()) {
                tempParams.append(String.format("%s=\"%s\"", key, paramsMap.get(key), "UTF-8"));
                if (pos < size - 1) {
                    tempParams.append("; ");
                }
                pos++;
            }
            tempParams.append("\r\n");
            tempParams.append("Content-Type: application/octet-stream\r\n");
            tempParams.append("\r\n");
            String params = tempParams.toString();
            requestStream.writeBytes(params);
            //发送文件数据
            FileInputStream fileInput = new FileInputStream(file);
            int bytesRead;
            byte[] buffer = new byte[1024];
            DataInputStream in = new DataInputStream(new FileInputStream(file));
            while ((bytesRead = in.read(buffer)) != -1) {
                requestStream.write(buffer, 0, bytesRead);
            }
            requestStream.writeBytes("\r\n");
            requestStream.flush();
            requestStream.writeBytes("--" + "*****" + "--" + "\r\n");
            requestStream.flush();
            fileInput.close();
            int statusCode = connection.getResponseCode();
            if (statusCode == 200) {
                // 获取返回的数据
                String result = streamToString(connection.getInputStream());
                Logger.i("upload success, result: " + result);
            } else {
                Logger.e("upload failure");
            }
        } catch (IOException e) {
            Logger.e(e.getMessage(), e);
        }
    }

    public static void failureCallback(final String tag, final Throwable t, final ResultCallback callback) {
        final Handler handler = new Handler(Looper.getMainLooper());
        handler.post(new Runnable() {
            @Override
            public void run() {
                callback.onFailure(tag, t);
            }
        });
    }

    public static void successCallback(final String tag, final Object o, final ResultCallback callback) {
        final Handler handler = new Handler(Looper.getMainLooper());
        handler.post(new Runnable() {
            @Override
            public void run() {
                callback.onResponse(tag, o);
            }
        });
    }

}
