/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.serverless.proxy.internal.servlet;

import com.amazonaws.serverless.proxy.internal.servlet.AwsHttpServletResponse;
import com.amazonaws.serverless.proxy.model.Headers;
import jakarta.servlet.http.Cookie;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.Calendar;
import java.util.Locale;
import java.util.TimeZone;
import java.util.concurrent.CountDownLatch;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class AwsHttpServletResponseTest {
    private static final int COOKIE_GRACE_COMPARE_MILLIS = 2000;
    private static final String COOKIE_NAME = "session_id";
    private static final String COOKIE_VALUE = "123";
    private static final String COOKIE_PATH = "/api";
    private static final String COOKIE_DOMAIN = "mydomain.com";
    private static final int MAX_AGE_VALUE = 300;
    private static final Pattern MAX_AGE_PATTERN = Pattern.compile("Max-Age=(-?[0-9]+)");
    private static final Pattern EXPIRES_PATTERN = Pattern.compile("Expires=([^;]+)");
    private static final String CONTENT_TYPE_WITH_CHARSET = "application/json; charset=UTF-8";
    private static final String JAVASCRIPT_CONTENT_TYPE_WITH_CHARSET = "application/javascript; charset=UTF-8";

    @Test
    void cookie_addCookie_verifyPath() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        Cookie pathCookie = new Cookie(COOKIE_NAME, COOKIE_VALUE);
        pathCookie.setPath(COOKIE_PATH);
        resp.addCookie(pathCookie);
        String cookieHeader = resp.getHeader("Set-Cookie");
        Assertions.assertNotNull((Object)cookieHeader);
        Assertions.assertTrue((boolean)cookieHeader.contains("Path=/api"));
        Assertions.assertTrue((boolean)cookieHeader.contains("session_id=123"));
    }

    @Test
    void cookie_addCookie_verifySecure() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        Cookie secureCookie = new Cookie(COOKIE_NAME, COOKIE_VALUE);
        secureCookie.setSecure(true);
        resp.addCookie(secureCookie);
        String cookieHeader = resp.getHeader("Set-Cookie");
        Assertions.assertNotNull((Object)cookieHeader);
        Assertions.assertTrue((boolean)cookieHeader.contains("; Secure"));
        Assertions.assertTrue((boolean)cookieHeader.contains("session_id=123"));
    }

    @Test
    void cookie_addCookie_verifyDomain() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        Cookie domainCookie = new Cookie(COOKIE_NAME, COOKIE_VALUE);
        domainCookie.setDomain(COOKIE_DOMAIN);
        resp.addCookie(domainCookie);
        String cookieHeader = resp.getHeader("Set-Cookie");
        Assertions.assertNotNull((Object)cookieHeader);
        Assertions.assertTrue((boolean)cookieHeader.contains("; Domain=mydomain.com"));
        Assertions.assertTrue((boolean)cookieHeader.contains("session_id=123"));
    }

    @Test
    void cookie_addCookie_defaultMaxAgeIsNegative() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        Cookie maxAgeCookie = new Cookie(COOKIE_NAME, COOKIE_VALUE);
        maxAgeCookie.setDomain(COOKIE_DOMAIN);
        resp.addCookie(maxAgeCookie);
        String cookieHeader = resp.getHeader("Set-Cookie");
        Assertions.assertNotNull((Object)cookieHeader);
        Assertions.assertFalse((boolean)cookieHeader.contains("Max-Age="));
        Assertions.assertTrue((boolean)cookieHeader.contains("session_id=123"));
    }

    @Test
    void cookie_addCookie_positiveMaxAgeIsPresent() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        Cookie maxAgeCookie = new Cookie(COOKIE_NAME, COOKIE_VALUE);
        maxAgeCookie.setMaxAge(300);
        resp.addCookie(maxAgeCookie);
        String cookieHeader = resp.getHeader("Set-Cookie");
        Assertions.assertNotNull((Object)cookieHeader);
        Assertions.assertTrue((boolean)cookieHeader.contains("; Max-Age="));
        Assertions.assertTrue((boolean)cookieHeader.contains("session_id=123"));
        int maxAge = this.getMaxAge(cookieHeader);
        Assertions.assertEquals((int)300, (int)maxAge);
    }

    @Test
    void cookie_addCookie_positiveMaxAgeExpiresDate() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        Cookie maxAgeCookie = new Cookie(COOKIE_NAME, COOKIE_VALUE);
        maxAgeCookie.setMaxAge(300);
        resp.addCookie(maxAgeCookie);
        Calendar testExpiration = Calendar.getInstance();
        testExpiration.add(13, 300);
        testExpiration.setTimeZone(TimeZone.getTimeZone("GMT"));
        String cookieHeader = resp.getHeader("Set-Cookie");
        Assertions.assertNotNull((Object)cookieHeader);
        Assertions.assertTrue((boolean)cookieHeader.contains("; Max-Age="));
        Assertions.assertTrue((boolean)cookieHeader.contains("session_id=123"));
        SimpleDateFormat dateFormat = new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss z");
        Calendar expiration = this.getExpires(cookieHeader);
        long dateDiff = testExpiration.getTimeInMillis() - expiration.getTimeInMillis();
        Assertions.assertTrue((Math.abs(dateDiff) < 2000L ? 1 : 0) != 0);
    }

    @Test
    void cookie_addCookieWithoutMaxAge_expectNoExpires() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        Cookie simpleCookie = new Cookie(COOKIE_NAME, COOKIE_VALUE);
        resp.addCookie(simpleCookie);
        String cookieHeader = resp.getHeader("Set-Cookie");
        Assertions.assertNotNull((Object)cookieHeader);
        Assertions.assertFalse((boolean)cookieHeader.contains("Expires"));
    }

    @Test
    void cookie_addCookieWithMaxAgeZero_expectExpiresInThePast() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        Cookie zeroMaxAgeCookie = new Cookie(COOKIE_NAME, COOKIE_VALUE);
        zeroMaxAgeCookie.setMaxAge(0);
        resp.addCookie(zeroMaxAgeCookie);
        String cookieHeader = resp.getHeader("Set-Cookie");
        Calendar cal = this.getExpires(cookieHeader);
        long currentTimeMillis = System.currentTimeMillis();
        Assertions.assertNotNull((Object)cookieHeader);
        Assertions.assertTrue((cal.getTimeInMillis() < currentTimeMillis ? 1 : 0) != 0);
        Assertions.assertTrue((boolean)cookieHeader.contains("session_id=123"));
    }

    @Test
    void responseHeaders_getAwsResponseHeaders_expectLatestHeader() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.addHeader("Content-Type", "application/json");
        resp.addHeader("content-type", "application/xml");
        Headers awsResp = resp.getAwsResponseHeaders();
        Assertions.assertEquals((int)1, (int)awsResp.size());
        Assertions.assertEquals((Object)"application/xml", (Object)awsResp.getFirst((Object)"Content-Type"));
    }

    @Test
    void responseHeaders_setHeaderWithNullValue_expectHeaderRemoved() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.setHeader("Content-Disposition", "inline");
        resp.setHeader("Content-Disposition", null);
        Headers awsResp = resp.getAwsResponseHeaders();
        Assertions.assertEquals((int)0, (int)awsResp.size());
    }

    @Test
    void responseHeaders_getAwsResponseHeaders_expectedMultpleCookieHeaders() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.addCookie(new Cookie(COOKIE_NAME, COOKIE_VALUE));
        resp.addCookie(new Cookie("Second", "test"));
        Headers awsResp = resp.getAwsResponseHeaders();
        Assertions.assertEquals((int)1, (int)awsResp.size());
        Assertions.assertEquals((int)2, (int)awsResp.get((Object)"Set-Cookie").size());
    }

    @Test
    void releaseLatch_flushBuffer_expectFlushToWriteAndRelease() {
        CountDownLatch respLatch = new CountDownLatch(1);
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, respLatch);
        String respBody = "Test resp";
        PrintWriter writer = null;
        try {
            PrintWriter finalWriter = writer = resp.getWriter();
            Runnable bodyWriter = () -> {
                finalWriter.write(respBody);
                try {
                    resp.flushBuffer();
                }
                catch (IOException e) {
                    Assertions.fail((String)"Could not flush buffer");
                }
            };
            new Thread(bodyWriter).start();
        }
        catch (IOException e) {
            Assertions.fail((String)"Could not get writer");
        }
        try {
            respLatch.await();
        }
        catch (InterruptedException e) {
            e.printStackTrace();
            Assertions.fail((String)"Response latch interrupted");
        }
        Assertions.assertEquals((long)0L, (long)respLatch.getCount());
        Assertions.assertNotNull((Object)writer);
        Assertions.assertEquals((Object)respBody, (Object)resp.getAwsResponseBodyString());
    }

    @Test
    void dateHeader_addDateHeader_expectMultipleHeaders() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.addDateHeader("Date", Instant.now().toEpochMilli());
        resp.addDateHeader("Date", Instant.now().toEpochMilli() - 1000L);
        Assertions.assertEquals((int)2, (int)resp.getHeaders("Date").size());
    }

    @Test
    void dateHeader_setDateHeader_expectSingleHeader() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.setDateHeader("Date", Instant.now().toEpochMilli());
        resp.setDateHeader("Date", Instant.now().toEpochMilli() - 1000L);
        Assertions.assertEquals((int)1, (int)resp.getHeaders("Date").size());
    }

    @Test
    void response_reset_expectEmptyHeadersAndBody() {
        CountDownLatch respLatch = new CountDownLatch(1);
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, respLatch);
        String body = "My Body";
        resp.addHeader("Test", "test");
        try {
            resp.getWriter().write(body);
            resp.flushBuffer();
        }
        catch (IOException e) {
            Assertions.fail((String)"Could not get writer");
        }
        Assertions.assertEquals((int)1, (int)resp.getHeaderNames().size());
        Assertions.assertEquals((Object)body, (Object)resp.getAwsResponseBodyString());
        resp.reset();
        Assertions.assertEquals((int)0, (int)resp.getHeaderNames().size());
    }

    @Test
    void headers_setIntHeader_expectSingleHeaderValue() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.setIntHeader("Test", 15);
        resp.setIntHeader("Test", 34);
        Assertions.assertEquals((int)1, (int)resp.getHeaderNames().size());
        Assertions.assertEquals((int)1, (int)resp.getHeaders("Test").size());
        Assertions.assertEquals((Object)"34", (Object)resp.getHeader("Test"));
    }

    @Test
    void headers_addIntHeader_expectMultipleHeaderValues() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.addIntHeader("Test", 15);
        resp.addIntHeader("Test", 34);
        Assertions.assertEquals((int)1, (int)resp.getHeaderNames().size());
        Assertions.assertEquals((int)2, (int)resp.getHeaders("Test").size());
        Assertions.assertEquals((Object)"15", (Object)resp.getHeader("Test"));
    }

    @Test
    void characterEncoding_setCharacterEncoding() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.setContentType("application/json");
        resp.setCharacterEncoding("UTF-8");
        Assertions.assertNotEquals((Object)"UTF-8", (Object)resp.getHeader("Content-Encoding"));
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getContentType());
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getHeader("Content-Type"));
    }

    @Test
    void characterEncoding_setContentType() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.setContentType("application/json; charset=utf-8");
        resp.setCharacterEncoding("UTF-8");
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getContentType());
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getHeader("Content-Type"));
        Assertions.assertEquals((Object)"UTF-8", (Object)resp.getCharacterEncoding());
    }

    @Test
    void characterEncoding_setContentTypeAndsetCharacterEncoding() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.setContentType("application/json");
        resp.setCharacterEncoding("UTF-8");
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getContentType());
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getHeader("Content-Type"));
        Assertions.assertEquals((Object)"UTF-8", (Object)resp.getCharacterEncoding());
    }

    @Test
    void characterEncoding_setCharacterEncodingAndsetContentType() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.setCharacterEncoding("UTF-8");
        resp.setContentType("application/json");
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getContentType());
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getHeader("Content-Type"));
        Assertions.assertEquals((Object)"UTF-8", (Object)resp.getCharacterEncoding());
    }

    @Test
    void characterEncoding_setCharacterEncodingInContentType_characterEncodingPopulatedCorrectly() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.setContentType(CONTENT_TYPE_WITH_CHARSET);
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getContentType());
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getHeader("Content-Type"));
        Assertions.assertEquals((Object)"UTF-8", (Object)resp.getCharacterEncoding());
    }

    @Test
    void characterEncoding_setCharacterEncodingInContentType_overridesDefault() {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, null);
        resp.setCharacterEncoding("ISO-8859-1");
        resp.setContentType(CONTENT_TYPE_WITH_CHARSET);
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getContentType());
        Assertions.assertEquals((Object)CONTENT_TYPE_WITH_CHARSET, (Object)resp.getHeader("Content-Type"));
        Assertions.assertEquals((Object)"UTF-8", (Object)resp.getCharacterEncoding());
    }

    @Test
    void characterEncoding_encodingInContentTypeHeader_writesCorrectData() throws IOException {
        AwsHttpServletResponse resp = new AwsHttpServletResponse(null, new CountDownLatch(1));
        resp.setHeader("Content-Type", JAVASCRIPT_CONTENT_TYPE_WITH_CHARSET);
        resp.getOutputStream().write("\u00fc".getBytes(StandardCharsets.UTF_8));
        resp.flushBuffer();
        Assertions.assertEquals((Object)JAVASCRIPT_CONTENT_TYPE_WITH_CHARSET, (Object)resp.getContentType());
        Assertions.assertEquals((Object)JAVASCRIPT_CONTENT_TYPE_WITH_CHARSET, (Object)resp.getHeader("Content-Type"));
        Assertions.assertEquals((Object)"\u00fc", (Object)resp.getAwsResponseBodyString());
    }

    private int getMaxAge(String header) {
        Matcher ageMatcher = MAX_AGE_PATTERN.matcher(header);
        Assertions.assertTrue((boolean)ageMatcher.find());
        Assertions.assertTrue((ageMatcher.groupCount() >= 1 ? 1 : 0) != 0);
        String ageString = ageMatcher.group(1);
        return Integer.parseInt(ageString);
    }

    private Calendar getExpires(String header) {
        Matcher ageMatcher = EXPIRES_PATTERN.matcher(header);
        Assertions.assertTrue((boolean)ageMatcher.find());
        Assertions.assertTrue((ageMatcher.groupCount() >= 1 ? 1 : 0) != 0);
        String expiresString = ageMatcher.group(1);
        SimpleDateFormat sdf = new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss z", Locale.US);
        Calendar cal = Calendar.getInstance();
        try {
            cal.setTime(sdf.parse(expiresString));
        }
        catch (ParseException e) {
            e.printStackTrace();
            Assertions.fail((String)"Could not parse expire date");
        }
        return cal;
    }
}

