/*
 * Copyright 2010-2023 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package com.amazonaws.services.kms.model;

import java.io.Serializable;

import com.amazonaws.AmazonWebServiceRequest;

/**
 * <p>
 * Verifies a digital signature that was generated by the <a>Sign</a> operation.
 * </p>
 * <p/>
 * <p>
 * Verification confirms that an authorized user signed the message with the
 * specified KMS key and signing algorithm, and the message hasn't changed since
 * it was signed. If the signature is verified, the value of the
 * <code>SignatureValid</code> field in the response is <code>True</code>. If
 * the signature verification fails, the <code>Verify</code> operation fails
 * with an <code>KMSInvalidSignatureException</code> exception.
 * </p>
 * <p>
 * A digital signature is generated by using the private key in an asymmetric
 * KMS key. The signature is verified by using the public key in the same
 * asymmetric KMS key. For information about asymmetric KMS keys, see <a href=
 * "https://docs.aws.amazon.com/kms/latest/developerguide/symmetric-asymmetric.html"
 * >Asymmetric KMS keys</a> in the <i>Key Management Service Developer
 * Guide</i>.
 * </p>
 * <p>
 * To use the <code>Verify</code> operation, specify the same asymmetric KMS
 * key, message, and signing algorithm that were used to produce the signature.
 * The message type does not need to be the same as the one used for signing,
 * but it must indicate whether the value of the <code>Message</code> parameter
 * should be hashed as part of the verification process.
 * </p>
 * <p>
 * You can also verify the digital signature by using the public key of the KMS
 * key outside of KMS. Use the <a>GetPublicKey</a> operation to download the
 * public key in the asymmetric KMS key and then use the public key to verify
 * the signature outside of KMS. The advantage of using the <code>Verify</code>
 * operation is that it is performed within KMS. As a result, it's easy to call,
 * the operation is performed within the FIPS boundary, it is logged in
 * CloudTrail, and you can use key policy and IAM policy to determine who is
 * authorized to use the KMS key to verify signatures.
 * </p>
 * <p>
 * To verify a signature outside of KMS with an SM2 public key (China Regions
 * only), you must specify the distinguishing ID. By default, KMS uses
 * <code>1234567812345678</code> as the distinguishing ID. For more information,
 * see <a href=
 * "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
 * >Offline verification with SM2 key pairs</a>.
 * </p>
 * <p>
 * The KMS key that you use for this operation must be in a compatible key
 * state. For details, see <a
 * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html"
 * >Key states of KMS keys</a> in the <i>Key Management Service Developer
 * Guide</i>.
 * </p>
 * <p>
 * <b>Cross-account use</b>: Yes. To perform this operation with a KMS key in a
 * different Amazon Web Services account, specify the key ARN or alias ARN in
 * the value of the <code>KeyId</code> parameter.
 * </p>
 * <p>
 * <b>Required permissions</b>: <a href=
 * "https://docs.aws.amazon.com/kms/latest/developerguide/kms-api-permissions-reference.html"
 * >kms:Verify</a> (key policy)
 * </p>
 * <p>
 * <b>Related operations</b>: <a>Sign</a>
 * </p>
 */
public class VerifyRequest extends AmazonWebServiceRequest implements Serializable {
    /**
     * <p>
     * Identifies the asymmetric KMS key that will be used to verify the
     * signature. This must be the same KMS key that was used to generate the
     * signature. If you specify a different KMS key, the signature verification
     * fails.
     * </p>
     * <p>
     * To specify a KMS key, use its key ID, key ARN, alias name, or alias ARN.
     * When using an alias name, prefix it with <code>"alias/"</code>. To
     * specify a KMS key in a different Amazon Web Services account, you must
     * use the key ARN or alias ARN.
     * </p>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN:
     * <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: <code>alias/ExampleAlias</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN:
     * <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get the key ID and key ARN for a KMS key, use <a>ListKeys</a> or
     * <a>DescribeKey</a>. To get the alias name and alias ARN, use
     * <a>ListAliases</a>.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 2048<br/>
     */
    private String keyId;

    /**
     * <p>
     * Specifies the message that was signed. You can submit a raw message of up
     * to 4096 bytes, or a hash digest of the message. If you submit a digest,
     * use the <code>MessageType</code> parameter with a value of
     * <code>DIGEST</code>.
     * </p>
     * <p>
     * If the message specified here is different from the message that was
     * signed, the signature verification fails. A message and its hash digest
     * are considered to be the same message.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 4096<br/>
     */
    private java.nio.ByteBuffer message;

    /**
     * <p>
     * Tells KMS whether the value of the <code>Message</code> parameter should
     * be hashed as part of the signing algorithm. Use <code>RAW</code> for
     * unhashed messages; use <code>DIGEST</code> for message digests, which are
     * already hashed.
     * </p>
     * <p>
     * When the value of <code>MessageType</code> is <code>RAW</code>, KMS uses
     * the standard signing algorithm, which begins with a hash function. When
     * the value is <code>DIGEST</code>, KMS skips the hashing step in the
     * signing algorithm.
     * </p>
     * <important>
     * <p>
     * Use the <code>DIGEST</code> value only when the value of the
     * <code>Message</code> parameter is a message digest. If you use the
     * <code>DIGEST</code> value with an unhashed message, the security of the
     * verification operation can be compromised.
     * </p>
     * </important>
     * <p>
     * When the value of <code>MessageType</code>is <code>DIGEST</code>, the
     * length of the <code>Message</code> value must match the length of hashed
     * messages for the specified signing algorithm.
     * </p>
     * <p>
     * You can submit a message digest and omit the <code>MessageType</code> or
     * specify <code>RAW</code> so the digest is hashed again while signing.
     * However, if the signed message is hashed once while signing, but twice
     * while verifying, verification fails, even when the message hasn't
     * changed.
     * </p>
     * <p>
     * The hashing algorithm in that <code>Verify</code> uses is based on the
     * <code>SigningAlgorithm</code> value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_256 use the SHA_256 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_384 use the SHA_384 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_512 use the SHA_512 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * SM2DSA uses the SM3 hashing algorithm. For details, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
     * >Offline verification with SM2 key pairs</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RAW, DIGEST
     */
    private String messageType;

    /**
     * <p>
     * The signature that the <code>Sign</code> operation generated.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 6144<br/>
     */
    private java.nio.ByteBuffer signature;

    /**
     * <p>
     * The signing algorithm that was used to sign the message. If you submit a
     * different algorithm, the signature verification fails.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RSASSA_PSS_SHA_256, RSASSA_PSS_SHA_384,
     * RSASSA_PSS_SHA_512, RSASSA_PKCS1_V1_5_SHA_256, RSASSA_PKCS1_V1_5_SHA_384,
     * RSASSA_PKCS1_V1_5_SHA_512, ECDSA_SHA_256, ECDSA_SHA_384, ECDSA_SHA_512,
     * SM2DSA
     */
    private String signingAlgorithm;

    /**
     * <p>
     * A list of grant tokens.
     * </p>
     * <p>
     * Use a grant token when your permission to call this operation comes from
     * a new grant that has not yet achieved <i>eventual consistency</i>. For
     * more information, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/grants.html#grant_token"
     * >Grant token</a> and <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/grant-manage.html#using-grant-token"
     * >Using a grant token</a> in the <i>Key Management Service Developer
     * Guide</i>.
     * </p>
     */
    private java.util.List<String> grantTokens = new java.util.ArrayList<String>();

    /**
     * <p>
     * Identifies the asymmetric KMS key that will be used to verify the
     * signature. This must be the same KMS key that was used to generate the
     * signature. If you specify a different KMS key, the signature verification
     * fails.
     * </p>
     * <p>
     * To specify a KMS key, use its key ID, key ARN, alias name, or alias ARN.
     * When using an alias name, prefix it with <code>"alias/"</code>. To
     * specify a KMS key in a different Amazon Web Services account, you must
     * use the key ARN or alias ARN.
     * </p>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN:
     * <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: <code>alias/ExampleAlias</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN:
     * <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get the key ID and key ARN for a KMS key, use <a>ListKeys</a> or
     * <a>DescribeKey</a>. To get the alias name and alias ARN, use
     * <a>ListAliases</a>.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 2048<br/>
     *
     * @return <p>
     *         Identifies the asymmetric KMS key that will be used to verify the
     *         signature. This must be the same KMS key that was used to
     *         generate the signature. If you specify a different KMS key, the
     *         signature verification fails.
     *         </p>
     *         <p>
     *         To specify a KMS key, use its key ID, key ARN, alias name, or
     *         alias ARN. When using an alias name, prefix it with
     *         <code>"alias/"</code>. To specify a KMS key in a different Amazon
     *         Web Services account, you must use the key ARN or alias ARN.
     *         </p>
     *         <p>
     *         For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN:
     *         <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name: <code>alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias ARN:
     *         <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         To get the key ID and key ARN for a KMS key, use <a>ListKeys</a>
     *         or <a>DescribeKey</a>. To get the alias name and alias ARN, use
     *         <a>ListAliases</a>.
     *         </p>
     */
    public String getKeyId() {
        return keyId;
    }

    /**
     * <p>
     * Identifies the asymmetric KMS key that will be used to verify the
     * signature. This must be the same KMS key that was used to generate the
     * signature. If you specify a different KMS key, the signature verification
     * fails.
     * </p>
     * <p>
     * To specify a KMS key, use its key ID, key ARN, alias name, or alias ARN.
     * When using an alias name, prefix it with <code>"alias/"</code>. To
     * specify a KMS key in a different Amazon Web Services account, you must
     * use the key ARN or alias ARN.
     * </p>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN:
     * <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: <code>alias/ExampleAlias</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN:
     * <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get the key ID and key ARN for a KMS key, use <a>ListKeys</a> or
     * <a>DescribeKey</a>. To get the alias name and alias ARN, use
     * <a>ListAliases</a>.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 2048<br/>
     *
     * @param keyId <p>
     *            Identifies the asymmetric KMS key that will be used to verify
     *            the signature. This must be the same KMS key that was used to
     *            generate the signature. If you specify a different KMS key,
     *            the signature verification fails.
     *            </p>
     *            <p>
     *            To specify a KMS key, use its key ID, key ARN, alias name, or
     *            alias ARN. When using an alias name, prefix it with
     *            <code>"alias/"</code>. To specify a KMS key in a different
     *            Amazon Web Services account, you must use the key ARN or alias
     *            ARN.
     *            </p>
     *            <p>
     *            For example:
     *            </p>
     *            <ul>
     *            <li>
     *            <p>
     *            Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Key ARN:
     *            <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Alias name: <code>alias/ExampleAlias</code>
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Alias ARN:
     *            <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     *            </p>
     *            </li>
     *            </ul>
     *            <p>
     *            To get the key ID and key ARN for a KMS key, use
     *            <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name
     *            and alias ARN, use <a>ListAliases</a>.
     *            </p>
     */
    public void setKeyId(String keyId) {
        this.keyId = keyId;
    }

    /**
     * <p>
     * Identifies the asymmetric KMS key that will be used to verify the
     * signature. This must be the same KMS key that was used to generate the
     * signature. If you specify a different KMS key, the signature verification
     * fails.
     * </p>
     * <p>
     * To specify a KMS key, use its key ID, key ARN, alias name, or alias ARN.
     * When using an alias name, prefix it with <code>"alias/"</code>. To
     * specify a KMS key in a different Amazon Web Services account, you must
     * use the key ARN or alias ARN.
     * </p>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN:
     * <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: <code>alias/ExampleAlias</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN:
     * <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get the key ID and key ARN for a KMS key, use <a>ListKeys</a> or
     * <a>DescribeKey</a>. To get the alias name and alias ARN, use
     * <a>ListAliases</a>.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 2048<br/>
     *
     * @param keyId <p>
     *            Identifies the asymmetric KMS key that will be used to verify
     *            the signature. This must be the same KMS key that was used to
     *            generate the signature. If you specify a different KMS key,
     *            the signature verification fails.
     *            </p>
     *            <p>
     *            To specify a KMS key, use its key ID, key ARN, alias name, or
     *            alias ARN. When using an alias name, prefix it with
     *            <code>"alias/"</code>. To specify a KMS key in a different
     *            Amazon Web Services account, you must use the key ARN or alias
     *            ARN.
     *            </p>
     *            <p>
     *            For example:
     *            </p>
     *            <ul>
     *            <li>
     *            <p>
     *            Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Key ARN:
     *            <code>arn:aws:kms:us-east-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Alias name: <code>alias/ExampleAlias</code>
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Alias ARN:
     *            <code>arn:aws:kms:us-east-2:111122223333:alias/ExampleAlias</code>
     *            </p>
     *            </li>
     *            </ul>
     *            <p>
     *            To get the key ID and key ARN for a KMS key, use
     *            <a>ListKeys</a> or <a>DescribeKey</a>. To get the alias name
     *            and alias ARN, use <a>ListAliases</a>.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public VerifyRequest withKeyId(String keyId) {
        this.keyId = keyId;
        return this;
    }

    /**
     * <p>
     * Specifies the message that was signed. You can submit a raw message of up
     * to 4096 bytes, or a hash digest of the message. If you submit a digest,
     * use the <code>MessageType</code> parameter with a value of
     * <code>DIGEST</code>.
     * </p>
     * <p>
     * If the message specified here is different from the message that was
     * signed, the signature verification fails. A message and its hash digest
     * are considered to be the same message.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 4096<br/>
     *
     * @return <p>
     *         Specifies the message that was signed. You can submit a raw
     *         message of up to 4096 bytes, or a hash digest of the message. If
     *         you submit a digest, use the <code>MessageType</code> parameter
     *         with a value of <code>DIGEST</code>.
     *         </p>
     *         <p>
     *         If the message specified here is different from the message that
     *         was signed, the signature verification fails. A message and its
     *         hash digest are considered to be the same message.
     *         </p>
     */
    public java.nio.ByteBuffer getMessage() {
        return message;
    }

    /**
     * <p>
     * Specifies the message that was signed. You can submit a raw message of up
     * to 4096 bytes, or a hash digest of the message. If you submit a digest,
     * use the <code>MessageType</code> parameter with a value of
     * <code>DIGEST</code>.
     * </p>
     * <p>
     * If the message specified here is different from the message that was
     * signed, the signature verification fails. A message and its hash digest
     * are considered to be the same message.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 4096<br/>
     *
     * @param message <p>
     *            Specifies the message that was signed. You can submit a raw
     *            message of up to 4096 bytes, or a hash digest of the message.
     *            If you submit a digest, use the <code>MessageType</code>
     *            parameter with a value of <code>DIGEST</code>.
     *            </p>
     *            <p>
     *            If the message specified here is different from the message
     *            that was signed, the signature verification fails. A message
     *            and its hash digest are considered to be the same message.
     *            </p>
     */
    public void setMessage(java.nio.ByteBuffer message) {
        this.message = message;
    }

    /**
     * <p>
     * Specifies the message that was signed. You can submit a raw message of up
     * to 4096 bytes, or a hash digest of the message. If you submit a digest,
     * use the <code>MessageType</code> parameter with a value of
     * <code>DIGEST</code>.
     * </p>
     * <p>
     * If the message specified here is different from the message that was
     * signed, the signature verification fails. A message and its hash digest
     * are considered to be the same message.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 4096<br/>
     *
     * @param message <p>
     *            Specifies the message that was signed. You can submit a raw
     *            message of up to 4096 bytes, or a hash digest of the message.
     *            If you submit a digest, use the <code>MessageType</code>
     *            parameter with a value of <code>DIGEST</code>.
     *            </p>
     *            <p>
     *            If the message specified here is different from the message
     *            that was signed, the signature verification fails. A message
     *            and its hash digest are considered to be the same message.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public VerifyRequest withMessage(java.nio.ByteBuffer message) {
        this.message = message;
        return this;
    }

    /**
     * <p>
     * Tells KMS whether the value of the <code>Message</code> parameter should
     * be hashed as part of the signing algorithm. Use <code>RAW</code> for
     * unhashed messages; use <code>DIGEST</code> for message digests, which are
     * already hashed.
     * </p>
     * <p>
     * When the value of <code>MessageType</code> is <code>RAW</code>, KMS uses
     * the standard signing algorithm, which begins with a hash function. When
     * the value is <code>DIGEST</code>, KMS skips the hashing step in the
     * signing algorithm.
     * </p>
     * <important>
     * <p>
     * Use the <code>DIGEST</code> value only when the value of the
     * <code>Message</code> parameter is a message digest. If you use the
     * <code>DIGEST</code> value with an unhashed message, the security of the
     * verification operation can be compromised.
     * </p>
     * </important>
     * <p>
     * When the value of <code>MessageType</code>is <code>DIGEST</code>, the
     * length of the <code>Message</code> value must match the length of hashed
     * messages for the specified signing algorithm.
     * </p>
     * <p>
     * You can submit a message digest and omit the <code>MessageType</code> or
     * specify <code>RAW</code> so the digest is hashed again while signing.
     * However, if the signed message is hashed once while signing, but twice
     * while verifying, verification fails, even when the message hasn't
     * changed.
     * </p>
     * <p>
     * The hashing algorithm in that <code>Verify</code> uses is based on the
     * <code>SigningAlgorithm</code> value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_256 use the SHA_256 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_384 use the SHA_384 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_512 use the SHA_512 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * SM2DSA uses the SM3 hashing algorithm. For details, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
     * >Offline verification with SM2 key pairs</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RAW, DIGEST
     *
     * @return <p>
     *         Tells KMS whether the value of the <code>Message</code> parameter
     *         should be hashed as part of the signing algorithm. Use
     *         <code>RAW</code> for unhashed messages; use <code>DIGEST</code>
     *         for message digests, which are already hashed.
     *         </p>
     *         <p>
     *         When the value of <code>MessageType</code> is <code>RAW</code>,
     *         KMS uses the standard signing algorithm, which begins with a hash
     *         function. When the value is <code>DIGEST</code>, KMS skips the
     *         hashing step in the signing algorithm.
     *         </p>
     *         <important>
     *         <p>
     *         Use the <code>DIGEST</code> value only when the value of the
     *         <code>Message</code> parameter is a message digest. If you use
     *         the <code>DIGEST</code> value with an unhashed message, the
     *         security of the verification operation can be compromised.
     *         </p>
     *         </important>
     *         <p>
     *         When the value of <code>MessageType</code>is <code>DIGEST</code>,
     *         the length of the <code>Message</code> value must match the
     *         length of hashed messages for the specified signing algorithm.
     *         </p>
     *         <p>
     *         You can submit a message digest and omit the
     *         <code>MessageType</code> or specify <code>RAW</code> so the
     *         digest is hashed again while signing. However, if the signed
     *         message is hashed once while signing, but twice while verifying,
     *         verification fails, even when the message hasn't changed.
     *         </p>
     *         <p>
     *         The hashing algorithm in that <code>Verify</code> uses is based
     *         on the <code>SigningAlgorithm</code> value.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Signing algorithms that end in SHA_256 use the SHA_256 hashing
     *         algorithm.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Signing algorithms that end in SHA_384 use the SHA_384 hashing
     *         algorithm.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Signing algorithms that end in SHA_512 use the SHA_512 hashing
     *         algorithm.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SM2DSA uses the SM3 hashing algorithm. For details, see <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
     *         >Offline verification with SM2 key pairs</a>.
     *         </p>
     *         </li>
     *         </ul>
     * @see MessageType
     */
    public String getMessageType() {
        return messageType;
    }

    /**
     * <p>
     * Tells KMS whether the value of the <code>Message</code> parameter should
     * be hashed as part of the signing algorithm. Use <code>RAW</code> for
     * unhashed messages; use <code>DIGEST</code> for message digests, which are
     * already hashed.
     * </p>
     * <p>
     * When the value of <code>MessageType</code> is <code>RAW</code>, KMS uses
     * the standard signing algorithm, which begins with a hash function. When
     * the value is <code>DIGEST</code>, KMS skips the hashing step in the
     * signing algorithm.
     * </p>
     * <important>
     * <p>
     * Use the <code>DIGEST</code> value only when the value of the
     * <code>Message</code> parameter is a message digest. If you use the
     * <code>DIGEST</code> value with an unhashed message, the security of the
     * verification operation can be compromised.
     * </p>
     * </important>
     * <p>
     * When the value of <code>MessageType</code>is <code>DIGEST</code>, the
     * length of the <code>Message</code> value must match the length of hashed
     * messages for the specified signing algorithm.
     * </p>
     * <p>
     * You can submit a message digest and omit the <code>MessageType</code> or
     * specify <code>RAW</code> so the digest is hashed again while signing.
     * However, if the signed message is hashed once while signing, but twice
     * while verifying, verification fails, even when the message hasn't
     * changed.
     * </p>
     * <p>
     * The hashing algorithm in that <code>Verify</code> uses is based on the
     * <code>SigningAlgorithm</code> value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_256 use the SHA_256 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_384 use the SHA_384 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_512 use the SHA_512 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * SM2DSA uses the SM3 hashing algorithm. For details, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
     * >Offline verification with SM2 key pairs</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RAW, DIGEST
     *
     * @param messageType <p>
     *            Tells KMS whether the value of the <code>Message</code>
     *            parameter should be hashed as part of the signing algorithm.
     *            Use <code>RAW</code> for unhashed messages; use
     *            <code>DIGEST</code> for message digests, which are already
     *            hashed.
     *            </p>
     *            <p>
     *            When the value of <code>MessageType</code> is <code>RAW</code>
     *            , KMS uses the standard signing algorithm, which begins with a
     *            hash function. When the value is <code>DIGEST</code>, KMS
     *            skips the hashing step in the signing algorithm.
     *            </p>
     *            <important>
     *            <p>
     *            Use the <code>DIGEST</code> value only when the value of the
     *            <code>Message</code> parameter is a message digest. If you use
     *            the <code>DIGEST</code> value with an unhashed message, the
     *            security of the verification operation can be compromised.
     *            </p>
     *            </important>
     *            <p>
     *            When the value of <code>MessageType</code>is
     *            <code>DIGEST</code>, the length of the <code>Message</code>
     *            value must match the length of hashed messages for the
     *            specified signing algorithm.
     *            </p>
     *            <p>
     *            You can submit a message digest and omit the
     *            <code>MessageType</code> or specify <code>RAW</code> so the
     *            digest is hashed again while signing. However, if the signed
     *            message is hashed once while signing, but twice while
     *            verifying, verification fails, even when the message hasn't
     *            changed.
     *            </p>
     *            <p>
     *            The hashing algorithm in that <code>Verify</code> uses is
     *            based on the <code>SigningAlgorithm</code> value.
     *            </p>
     *            <ul>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_256 use the SHA_256 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_384 use the SHA_384 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_512 use the SHA_512 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            SM2DSA uses the SM3 hashing algorithm. For details, see <a
     *            href=
     *            "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
     *            >Offline verification with SM2 key pairs</a>.
     *            </p>
     *            </li>
     *            </ul>
     * @see MessageType
     */
    public void setMessageType(String messageType) {
        this.messageType = messageType;
    }

    /**
     * <p>
     * Tells KMS whether the value of the <code>Message</code> parameter should
     * be hashed as part of the signing algorithm. Use <code>RAW</code> for
     * unhashed messages; use <code>DIGEST</code> for message digests, which are
     * already hashed.
     * </p>
     * <p>
     * When the value of <code>MessageType</code> is <code>RAW</code>, KMS uses
     * the standard signing algorithm, which begins with a hash function. When
     * the value is <code>DIGEST</code>, KMS skips the hashing step in the
     * signing algorithm.
     * </p>
     * <important>
     * <p>
     * Use the <code>DIGEST</code> value only when the value of the
     * <code>Message</code> parameter is a message digest. If you use the
     * <code>DIGEST</code> value with an unhashed message, the security of the
     * verification operation can be compromised.
     * </p>
     * </important>
     * <p>
     * When the value of <code>MessageType</code>is <code>DIGEST</code>, the
     * length of the <code>Message</code> value must match the length of hashed
     * messages for the specified signing algorithm.
     * </p>
     * <p>
     * You can submit a message digest and omit the <code>MessageType</code> or
     * specify <code>RAW</code> so the digest is hashed again while signing.
     * However, if the signed message is hashed once while signing, but twice
     * while verifying, verification fails, even when the message hasn't
     * changed.
     * </p>
     * <p>
     * The hashing algorithm in that <code>Verify</code> uses is based on the
     * <code>SigningAlgorithm</code> value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_256 use the SHA_256 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_384 use the SHA_384 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_512 use the SHA_512 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * SM2DSA uses the SM3 hashing algorithm. For details, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
     * >Offline verification with SM2 key pairs</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RAW, DIGEST
     *
     * @param messageType <p>
     *            Tells KMS whether the value of the <code>Message</code>
     *            parameter should be hashed as part of the signing algorithm.
     *            Use <code>RAW</code> for unhashed messages; use
     *            <code>DIGEST</code> for message digests, which are already
     *            hashed.
     *            </p>
     *            <p>
     *            When the value of <code>MessageType</code> is <code>RAW</code>
     *            , KMS uses the standard signing algorithm, which begins with a
     *            hash function. When the value is <code>DIGEST</code>, KMS
     *            skips the hashing step in the signing algorithm.
     *            </p>
     *            <important>
     *            <p>
     *            Use the <code>DIGEST</code> value only when the value of the
     *            <code>Message</code> parameter is a message digest. If you use
     *            the <code>DIGEST</code> value with an unhashed message, the
     *            security of the verification operation can be compromised.
     *            </p>
     *            </important>
     *            <p>
     *            When the value of <code>MessageType</code>is
     *            <code>DIGEST</code>, the length of the <code>Message</code>
     *            value must match the length of hashed messages for the
     *            specified signing algorithm.
     *            </p>
     *            <p>
     *            You can submit a message digest and omit the
     *            <code>MessageType</code> or specify <code>RAW</code> so the
     *            digest is hashed again while signing. However, if the signed
     *            message is hashed once while signing, but twice while
     *            verifying, verification fails, even when the message hasn't
     *            changed.
     *            </p>
     *            <p>
     *            The hashing algorithm in that <code>Verify</code> uses is
     *            based on the <code>SigningAlgorithm</code> value.
     *            </p>
     *            <ul>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_256 use the SHA_256 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_384 use the SHA_384 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_512 use the SHA_512 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            SM2DSA uses the SM3 hashing algorithm. For details, see <a
     *            href=
     *            "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
     *            >Offline verification with SM2 key pairs</a>.
     *            </p>
     *            </li>
     *            </ul>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     * @see MessageType
     */
    public VerifyRequest withMessageType(String messageType) {
        this.messageType = messageType;
        return this;
    }

    /**
     * <p>
     * Tells KMS whether the value of the <code>Message</code> parameter should
     * be hashed as part of the signing algorithm. Use <code>RAW</code> for
     * unhashed messages; use <code>DIGEST</code> for message digests, which are
     * already hashed.
     * </p>
     * <p>
     * When the value of <code>MessageType</code> is <code>RAW</code>, KMS uses
     * the standard signing algorithm, which begins with a hash function. When
     * the value is <code>DIGEST</code>, KMS skips the hashing step in the
     * signing algorithm.
     * </p>
     * <important>
     * <p>
     * Use the <code>DIGEST</code> value only when the value of the
     * <code>Message</code> parameter is a message digest. If you use the
     * <code>DIGEST</code> value with an unhashed message, the security of the
     * verification operation can be compromised.
     * </p>
     * </important>
     * <p>
     * When the value of <code>MessageType</code>is <code>DIGEST</code>, the
     * length of the <code>Message</code> value must match the length of hashed
     * messages for the specified signing algorithm.
     * </p>
     * <p>
     * You can submit a message digest and omit the <code>MessageType</code> or
     * specify <code>RAW</code> so the digest is hashed again while signing.
     * However, if the signed message is hashed once while signing, but twice
     * while verifying, verification fails, even when the message hasn't
     * changed.
     * </p>
     * <p>
     * The hashing algorithm in that <code>Verify</code> uses is based on the
     * <code>SigningAlgorithm</code> value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_256 use the SHA_256 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_384 use the SHA_384 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_512 use the SHA_512 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * SM2DSA uses the SM3 hashing algorithm. For details, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
     * >Offline verification with SM2 key pairs</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RAW, DIGEST
     *
     * @param messageType <p>
     *            Tells KMS whether the value of the <code>Message</code>
     *            parameter should be hashed as part of the signing algorithm.
     *            Use <code>RAW</code> for unhashed messages; use
     *            <code>DIGEST</code> for message digests, which are already
     *            hashed.
     *            </p>
     *            <p>
     *            When the value of <code>MessageType</code> is <code>RAW</code>
     *            , KMS uses the standard signing algorithm, which begins with a
     *            hash function. When the value is <code>DIGEST</code>, KMS
     *            skips the hashing step in the signing algorithm.
     *            </p>
     *            <important>
     *            <p>
     *            Use the <code>DIGEST</code> value only when the value of the
     *            <code>Message</code> parameter is a message digest. If you use
     *            the <code>DIGEST</code> value with an unhashed message, the
     *            security of the verification operation can be compromised.
     *            </p>
     *            </important>
     *            <p>
     *            When the value of <code>MessageType</code>is
     *            <code>DIGEST</code>, the length of the <code>Message</code>
     *            value must match the length of hashed messages for the
     *            specified signing algorithm.
     *            </p>
     *            <p>
     *            You can submit a message digest and omit the
     *            <code>MessageType</code> or specify <code>RAW</code> so the
     *            digest is hashed again while signing. However, if the signed
     *            message is hashed once while signing, but twice while
     *            verifying, verification fails, even when the message hasn't
     *            changed.
     *            </p>
     *            <p>
     *            The hashing algorithm in that <code>Verify</code> uses is
     *            based on the <code>SigningAlgorithm</code> value.
     *            </p>
     *            <ul>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_256 use the SHA_256 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_384 use the SHA_384 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_512 use the SHA_512 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            SM2DSA uses the SM3 hashing algorithm. For details, see <a
     *            href=
     *            "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
     *            >Offline verification with SM2 key pairs</a>.
     *            </p>
     *            </li>
     *            </ul>
     * @see MessageType
     */
    public void setMessageType(MessageType messageType) {
        this.messageType = messageType.toString();
    }

    /**
     * <p>
     * Tells KMS whether the value of the <code>Message</code> parameter should
     * be hashed as part of the signing algorithm. Use <code>RAW</code> for
     * unhashed messages; use <code>DIGEST</code> for message digests, which are
     * already hashed.
     * </p>
     * <p>
     * When the value of <code>MessageType</code> is <code>RAW</code>, KMS uses
     * the standard signing algorithm, which begins with a hash function. When
     * the value is <code>DIGEST</code>, KMS skips the hashing step in the
     * signing algorithm.
     * </p>
     * <important>
     * <p>
     * Use the <code>DIGEST</code> value only when the value of the
     * <code>Message</code> parameter is a message digest. If you use the
     * <code>DIGEST</code> value with an unhashed message, the security of the
     * verification operation can be compromised.
     * </p>
     * </important>
     * <p>
     * When the value of <code>MessageType</code>is <code>DIGEST</code>, the
     * length of the <code>Message</code> value must match the length of hashed
     * messages for the specified signing algorithm.
     * </p>
     * <p>
     * You can submit a message digest and omit the <code>MessageType</code> or
     * specify <code>RAW</code> so the digest is hashed again while signing.
     * However, if the signed message is hashed once while signing, but twice
     * while verifying, verification fails, even when the message hasn't
     * changed.
     * </p>
     * <p>
     * The hashing algorithm in that <code>Verify</code> uses is based on the
     * <code>SigningAlgorithm</code> value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_256 use the SHA_256 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_384 use the SHA_384 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * Signing algorithms that end in SHA_512 use the SHA_512 hashing algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * SM2DSA uses the SM3 hashing algorithm. For details, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
     * >Offline verification with SM2 key pairs</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RAW, DIGEST
     *
     * @param messageType <p>
     *            Tells KMS whether the value of the <code>Message</code>
     *            parameter should be hashed as part of the signing algorithm.
     *            Use <code>RAW</code> for unhashed messages; use
     *            <code>DIGEST</code> for message digests, which are already
     *            hashed.
     *            </p>
     *            <p>
     *            When the value of <code>MessageType</code> is <code>RAW</code>
     *            , KMS uses the standard signing algorithm, which begins with a
     *            hash function. When the value is <code>DIGEST</code>, KMS
     *            skips the hashing step in the signing algorithm.
     *            </p>
     *            <important>
     *            <p>
     *            Use the <code>DIGEST</code> value only when the value of the
     *            <code>Message</code> parameter is a message digest. If you use
     *            the <code>DIGEST</code> value with an unhashed message, the
     *            security of the verification operation can be compromised.
     *            </p>
     *            </important>
     *            <p>
     *            When the value of <code>MessageType</code>is
     *            <code>DIGEST</code>, the length of the <code>Message</code>
     *            value must match the length of hashed messages for the
     *            specified signing algorithm.
     *            </p>
     *            <p>
     *            You can submit a message digest and omit the
     *            <code>MessageType</code> or specify <code>RAW</code> so the
     *            digest is hashed again while signing. However, if the signed
     *            message is hashed once while signing, but twice while
     *            verifying, verification fails, even when the message hasn't
     *            changed.
     *            </p>
     *            <p>
     *            The hashing algorithm in that <code>Verify</code> uses is
     *            based on the <code>SigningAlgorithm</code> value.
     *            </p>
     *            <ul>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_256 use the SHA_256 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_384 use the SHA_384 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            Signing algorithms that end in SHA_512 use the SHA_512 hashing
     *            algorithm.
     *            </p>
     *            </li>
     *            <li>
     *            <p>
     *            SM2DSA uses the SM3 hashing algorithm. For details, see <a
     *            href=
     *            "https://docs.aws.amazon.com/kms/latest/developerguide/asymmetric-key-specs.html#key-spec-sm-offline-verification"
     *            >Offline verification with SM2 key pairs</a>.
     *            </p>
     *            </li>
     *            </ul>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     * @see MessageType
     */
    public VerifyRequest withMessageType(MessageType messageType) {
        this.messageType = messageType.toString();
        return this;
    }

    /**
     * <p>
     * The signature that the <code>Sign</code> operation generated.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 6144<br/>
     *
     * @return <p>
     *         The signature that the <code>Sign</code> operation generated.
     *         </p>
     */
    public java.nio.ByteBuffer getSignature() {
        return signature;
    }

    /**
     * <p>
     * The signature that the <code>Sign</code> operation generated.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 6144<br/>
     *
     * @param signature <p>
     *            The signature that the <code>Sign</code> operation generated.
     *            </p>
     */
    public void setSignature(java.nio.ByteBuffer signature) {
        this.signature = signature;
    }

    /**
     * <p>
     * The signature that the <code>Sign</code> operation generated.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Length: </b>1 - 6144<br/>
     *
     * @param signature <p>
     *            The signature that the <code>Sign</code> operation generated.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public VerifyRequest withSignature(java.nio.ByteBuffer signature) {
        this.signature = signature;
        return this;
    }

    /**
     * <p>
     * The signing algorithm that was used to sign the message. If you submit a
     * different algorithm, the signature verification fails.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RSASSA_PSS_SHA_256, RSASSA_PSS_SHA_384,
     * RSASSA_PSS_SHA_512, RSASSA_PKCS1_V1_5_SHA_256, RSASSA_PKCS1_V1_5_SHA_384,
     * RSASSA_PKCS1_V1_5_SHA_512, ECDSA_SHA_256, ECDSA_SHA_384, ECDSA_SHA_512,
     * SM2DSA
     *
     * @return <p>
     *         The signing algorithm that was used to sign the message. If you
     *         submit a different algorithm, the signature verification fails.
     *         </p>
     * @see SigningAlgorithmSpec
     */
    public String getSigningAlgorithm() {
        return signingAlgorithm;
    }

    /**
     * <p>
     * The signing algorithm that was used to sign the message. If you submit a
     * different algorithm, the signature verification fails.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RSASSA_PSS_SHA_256, RSASSA_PSS_SHA_384,
     * RSASSA_PSS_SHA_512, RSASSA_PKCS1_V1_5_SHA_256, RSASSA_PKCS1_V1_5_SHA_384,
     * RSASSA_PKCS1_V1_5_SHA_512, ECDSA_SHA_256, ECDSA_SHA_384, ECDSA_SHA_512,
     * SM2DSA
     *
     * @param signingAlgorithm <p>
     *            The signing algorithm that was used to sign the message. If
     *            you submit a different algorithm, the signature verification
     *            fails.
     *            </p>
     * @see SigningAlgorithmSpec
     */
    public void setSigningAlgorithm(String signingAlgorithm) {
        this.signingAlgorithm = signingAlgorithm;
    }

    /**
     * <p>
     * The signing algorithm that was used to sign the message. If you submit a
     * different algorithm, the signature verification fails.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RSASSA_PSS_SHA_256, RSASSA_PSS_SHA_384,
     * RSASSA_PSS_SHA_512, RSASSA_PKCS1_V1_5_SHA_256, RSASSA_PKCS1_V1_5_SHA_384,
     * RSASSA_PKCS1_V1_5_SHA_512, ECDSA_SHA_256, ECDSA_SHA_384, ECDSA_SHA_512,
     * SM2DSA
     *
     * @param signingAlgorithm <p>
     *            The signing algorithm that was used to sign the message. If
     *            you submit a different algorithm, the signature verification
     *            fails.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     * @see SigningAlgorithmSpec
     */
    public VerifyRequest withSigningAlgorithm(String signingAlgorithm) {
        this.signingAlgorithm = signingAlgorithm;
        return this;
    }

    /**
     * <p>
     * The signing algorithm that was used to sign the message. If you submit a
     * different algorithm, the signature verification fails.
     * </p>
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RSASSA_PSS_SHA_256, RSASSA_PSS_SHA_384,
     * RSASSA_PSS_SHA_512, RSASSA_PKCS1_V1_5_SHA_256, RSASSA_PKCS1_V1_5_SHA_384,
     * RSASSA_PKCS1_V1_5_SHA_512, ECDSA_SHA_256, ECDSA_SHA_384, ECDSA_SHA_512,
     * SM2DSA
     *
     * @param signingAlgorithm <p>
     *            The signing algorithm that was used to sign the message. If
     *            you submit a different algorithm, the signature verification
     *            fails.
     *            </p>
     * @see SigningAlgorithmSpec
     */
    public void setSigningAlgorithm(SigningAlgorithmSpec signingAlgorithm) {
        this.signingAlgorithm = signingAlgorithm.toString();
    }

    /**
     * <p>
     * The signing algorithm that was used to sign the message. If you submit a
     * different algorithm, the signature verification fails.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     * <p>
     * <b>Constraints:</b><br/>
     * <b>Allowed Values: </b>RSASSA_PSS_SHA_256, RSASSA_PSS_SHA_384,
     * RSASSA_PSS_SHA_512, RSASSA_PKCS1_V1_5_SHA_256, RSASSA_PKCS1_V1_5_SHA_384,
     * RSASSA_PKCS1_V1_5_SHA_512, ECDSA_SHA_256, ECDSA_SHA_384, ECDSA_SHA_512,
     * SM2DSA
     *
     * @param signingAlgorithm <p>
     *            The signing algorithm that was used to sign the message. If
     *            you submit a different algorithm, the signature verification
     *            fails.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     * @see SigningAlgorithmSpec
     */
    public VerifyRequest withSigningAlgorithm(SigningAlgorithmSpec signingAlgorithm) {
        this.signingAlgorithm = signingAlgorithm.toString();
        return this;
    }

    /**
     * <p>
     * A list of grant tokens.
     * </p>
     * <p>
     * Use a grant token when your permission to call this operation comes from
     * a new grant that has not yet achieved <i>eventual consistency</i>. For
     * more information, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/grants.html#grant_token"
     * >Grant token</a> and <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/grant-manage.html#using-grant-token"
     * >Using a grant token</a> in the <i>Key Management Service Developer
     * Guide</i>.
     * </p>
     *
     * @return <p>
     *         A list of grant tokens.
     *         </p>
     *         <p>
     *         Use a grant token when your permission to call this operation
     *         comes from a new grant that has not yet achieved <i>eventual
     *         consistency</i>. For more information, see <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/grants.html#grant_token"
     *         >Grant token</a> and <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/grant-manage.html#using-grant-token"
     *         >Using a grant token</a> in the <i>Key Management Service
     *         Developer Guide</i>.
     *         </p>
     */
    public java.util.List<String> getGrantTokens() {
        return grantTokens;
    }

    /**
     * <p>
     * A list of grant tokens.
     * </p>
     * <p>
     * Use a grant token when your permission to call this operation comes from
     * a new grant that has not yet achieved <i>eventual consistency</i>. For
     * more information, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/grants.html#grant_token"
     * >Grant token</a> and <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/grant-manage.html#using-grant-token"
     * >Using a grant token</a> in the <i>Key Management Service Developer
     * Guide</i>.
     * </p>
     *
     * @param grantTokens <p>
     *            A list of grant tokens.
     *            </p>
     *            <p>
     *            Use a grant token when your permission to call this operation
     *            comes from a new grant that has not yet achieved <i>eventual
     *            consistency</i>. For more information, see <a href=
     *            "https://docs.aws.amazon.com/kms/latest/developerguide/grants.html#grant_token"
     *            >Grant token</a> and <a href=
     *            "https://docs.aws.amazon.com/kms/latest/developerguide/grant-manage.html#using-grant-token"
     *            >Using a grant token</a> in the <i>Key Management Service
     *            Developer Guide</i>.
     *            </p>
     */
    public void setGrantTokens(java.util.Collection<String> grantTokens) {
        if (grantTokens == null) {
            this.grantTokens = null;
            return;
        }

        this.grantTokens = new java.util.ArrayList<String>(grantTokens);
    }

    /**
     * <p>
     * A list of grant tokens.
     * </p>
     * <p>
     * Use a grant token when your permission to call this operation comes from
     * a new grant that has not yet achieved <i>eventual consistency</i>. For
     * more information, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/grants.html#grant_token"
     * >Grant token</a> and <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/grant-manage.html#using-grant-token"
     * >Using a grant token</a> in the <i>Key Management Service Developer
     * Guide</i>.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     *
     * @param grantTokens <p>
     *            A list of grant tokens.
     *            </p>
     *            <p>
     *            Use a grant token when your permission to call this operation
     *            comes from a new grant that has not yet achieved <i>eventual
     *            consistency</i>. For more information, see <a href=
     *            "https://docs.aws.amazon.com/kms/latest/developerguide/grants.html#grant_token"
     *            >Grant token</a> and <a href=
     *            "https://docs.aws.amazon.com/kms/latest/developerguide/grant-manage.html#using-grant-token"
     *            >Using a grant token</a> in the <i>Key Management Service
     *            Developer Guide</i>.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public VerifyRequest withGrantTokens(String... grantTokens) {
        if (getGrantTokens() == null) {
            this.grantTokens = new java.util.ArrayList<String>(grantTokens.length);
        }
        for (String value : grantTokens) {
            this.grantTokens.add(value);
        }
        return this;
    }

    /**
     * <p>
     * A list of grant tokens.
     * </p>
     * <p>
     * Use a grant token when your permission to call this operation comes from
     * a new grant that has not yet achieved <i>eventual consistency</i>. For
     * more information, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/grants.html#grant_token"
     * >Grant token</a> and <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/grant-manage.html#using-grant-token"
     * >Using a grant token</a> in the <i>Key Management Service Developer
     * Guide</i>.
     * </p>
     * <p>
     * Returns a reference to this object so that method calls can be chained
     * together.
     *
     * @param grantTokens <p>
     *            A list of grant tokens.
     *            </p>
     *            <p>
     *            Use a grant token when your permission to call this operation
     *            comes from a new grant that has not yet achieved <i>eventual
     *            consistency</i>. For more information, see <a href=
     *            "https://docs.aws.amazon.com/kms/latest/developerguide/grants.html#grant_token"
     *            >Grant token</a> and <a href=
     *            "https://docs.aws.amazon.com/kms/latest/developerguide/grant-manage.html#using-grant-token"
     *            >Using a grant token</a> in the <i>Key Management Service
     *            Developer Guide</i>.
     *            </p>
     * @return A reference to this updated object so that method calls can be
     *         chained together.
     */
    public VerifyRequest withGrantTokens(java.util.Collection<String> grantTokens) {
        setGrantTokens(grantTokens);
        return this;
    }

    /**
     * Returns a string representation of this object; useful for testing and
     * debugging.
     *
     * @return A string representation of this object.
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (getKeyId() != null)
            sb.append("KeyId: " + getKeyId() + ",");
        if (getMessage() != null)
            sb.append("Message: " + getMessage() + ",");
        if (getMessageType() != null)
            sb.append("MessageType: " + getMessageType() + ",");
        if (getSignature() != null)
            sb.append("Signature: " + getSignature() + ",");
        if (getSigningAlgorithm() != null)
            sb.append("SigningAlgorithm: " + getSigningAlgorithm() + ",");
        if (getGrantTokens() != null)
            sb.append("GrantTokens: " + getGrantTokens());
        sb.append("}");
        return sb.toString();
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int hashCode = 1;

        hashCode = prime * hashCode + ((getKeyId() == null) ? 0 : getKeyId().hashCode());
        hashCode = prime * hashCode + ((getMessage() == null) ? 0 : getMessage().hashCode());
        hashCode = prime * hashCode
                + ((getMessageType() == null) ? 0 : getMessageType().hashCode());
        hashCode = prime * hashCode + ((getSignature() == null) ? 0 : getSignature().hashCode());
        hashCode = prime * hashCode
                + ((getSigningAlgorithm() == null) ? 0 : getSigningAlgorithm().hashCode());
        hashCode = prime * hashCode
                + ((getGrantTokens() == null) ? 0 : getGrantTokens().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;

        if (obj instanceof VerifyRequest == false)
            return false;
        VerifyRequest other = (VerifyRequest) obj;

        if (other.getKeyId() == null ^ this.getKeyId() == null)
            return false;
        if (other.getKeyId() != null && other.getKeyId().equals(this.getKeyId()) == false)
            return false;
        if (other.getMessage() == null ^ this.getMessage() == null)
            return false;
        if (other.getMessage() != null && other.getMessage().equals(this.getMessage()) == false)
            return false;
        if (other.getMessageType() == null ^ this.getMessageType() == null)
            return false;
        if (other.getMessageType() != null
                && other.getMessageType().equals(this.getMessageType()) == false)
            return false;
        if (other.getSignature() == null ^ this.getSignature() == null)
            return false;
        if (other.getSignature() != null
                && other.getSignature().equals(this.getSignature()) == false)
            return false;
        if (other.getSigningAlgorithm() == null ^ this.getSigningAlgorithm() == null)
            return false;
        if (other.getSigningAlgorithm() != null
                && other.getSigningAlgorithm().equals(this.getSigningAlgorithm()) == false)
            return false;
        if (other.getGrantTokens() == null ^ this.getGrantTokens() == null)
            return false;
        if (other.getGrantTokens() != null
                && other.getGrantTokens().equals(this.getGrantTokens()) == false)
            return false;
        return true;
    }
}
