/**
 * Copyright 2017-2018 Amazon.com,
 * Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Amazon Software License (the "License").
 * You may not use this file except in compliance with the
 * License. A copy of the License is located at
 *
 *     http://aws.amazon.com/asl/
 *
 * or in the "license" file accompanying this file. This file is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, express or implied. See the License
 * for the specific language governing permissions and
 * limitations under the License.
 */

package com.amazonaws.kinesisvideo.producer;

import static java.util.Objects.requireNonNull;

import java.nio.ByteBuffer;

import android.support.annotation.NonNull;

/**
 * Kinesis Video frame representation.
 *
 * NOTE: This class must match the Frame declaration in native code in
 * /mkvgen/Include.h
 */

public class KinesisVideoFrame {
    /**
     * Index of the frame
     */
    private final int mIndex;

    /**
     * Frame flags
     */
    private final int mFlags;

    /**
     * The decoding timestamp of the frame in 100ns precision
     */
    private final long mDecodingTs;

    /**
     * The presentation timestamp of the frame in 100ns precision
     */
    private final long mPresentationTs;

    /**
     * The duration of the frame in 100ns precision
     */
    private final long mDuration;

    /**
     * The actual frame data
     */
    private final ByteBuffer mData;

    public KinesisVideoFrame(int index, int flags, long decodingTs, long presentationTs, long duration,
            @NonNull ByteBuffer data) {
        mIndex = index;
        mFlags = flags;
        mDecodingTs = decodingTs;
        mPresentationTs = presentationTs;
        mDuration = duration;
        mData = requireNonNull(data);
        // In some devices encoder would generate frames with more than 3 trailing zeros
        // which is not allowed by AnnexB specification
        removeTrailingZeros();
    }

    public int getIndex() {
        return mIndex;
    }

    public int getFlags() {
        return mFlags;
    }

    public long getDecodingTs() {
        return mDecodingTs;
    }

    public long getPresentationTs() {
        return mPresentationTs;
    }

    public long getDuration() {
        return mDuration;
    }

    public int getSize() {
        return mData.remaining();
    }

    @NonNull
    public ByteBuffer getData() {
        ByteBuffer byteBuffer = mData;
        try {
            if (mData.hasArray()) {
                byteBuffer = ByteBuffer.allocateDirect(mData.remaining());
                byteBuffer.put(mData);
            }
        } catch(final Exception e) {
            // Some Android implementations throw when accessing hasArray() API. We will ignore it
        }

        return byteBuffer;
    }

    @Override public String toString() {
        return new StringBuilder().append(getClass().getSimpleName()).append("{").append("mIndex=").append(mIndex)
                .append(", mFlags=").append(mFlags).append(", mDecodingTs=").append(mDecodingTs)
                .append(", mPresentationTs=").append(mPresentationTs).append(", mDuration=").append(mDuration)
                .append(", mData=").append(mData).append("}").toString();
    }

    private void removeTrailingZeros() {
        for (int index = mData.limit() - 1; index > mData.position(); index--) {
            if (mData.get(index) != 0) {
                mData.limit(index + 1);
                break;
            }
        }
    }
}
