/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.internal.keyvaluestore;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import com.amazonaws.internal.keyvaluestore.KeyNotFoundException;
import com.amazonaws.internal.keyvaluestore.KeyNotGeneratedException;
import com.amazonaws.internal.keyvaluestore.KeyProvider;
import com.amazonaws.internal.keyvaluestore.KeyProvider10;
import com.amazonaws.internal.keyvaluestore.KeyProvider18;
import com.amazonaws.internal.keyvaluestore.KeyProvider23;
import com.amazonaws.logging.Log;
import com.amazonaws.logging.LogFactory;
import com.amazonaws.util.Base64;
import java.security.Key;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import javax.crypto.Cipher;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.IvParameterSpec;

public class AWSKeyValueStore {
    private static final Log logger = LogFactory.getLog(AWSKeyValueStore.class);
    static Map<String, HashMap<String, String>> cacheFactory = new HashMap<String, HashMap<String, String>>();
    private Map<String, String> cache;
    private boolean isPersistenceEnabled;
    Context context;
    SharedPreferences sharedPreferencesForData;
    private final String sharedPreferencesName;
    SharedPreferences sharedPreferencesForEncryptionMaterials;
    KeyProvider keyProvider;
    private SecureRandom secureRandom = new SecureRandom();
    private static final String CIPHER_AES_GCM_NOPADDING = "AES/GCM/NoPadding";
    private static final int CIPHER_AES_GCM_NOPADDING_IV_LENGTH_IN_BYTES = 12;
    private static final int CIPHER_AES_GCM_NOPADDING_TAG_LENGTH_LENGTH_IN_BITS = 128;
    private static final String CHARSET_NAME = "UTF-8";
    static final String SHARED_PREFERENCES_DATA_IDENTIFIER_SUFFIX = ".encrypted";
    static final String SHARED_PREFERENCES_IV_SUFFIX = ".iv";
    static final String SHARED_PREFERENCES_STORE_VERSION_SUFFIX = ".keyvaluestoreversion";
    static final String SHARED_PREFERENCES_ENCRYPTION_KEY_NAMESPACE_SUFFIX = ".encryptionkey";
    private static final int AWS_KEY_VALUE_STORE_VERSION = 1;

    private static Map<String, String> getCacheForKey(String key) {
        if (cacheFactory.containsKey(key)) {
            return cacheFactory.get(key);
        }
        HashMap<String, String> cache = new HashMap<String, String>();
        cacheFactory.put(key, cache);
        return cache;
    }

    public AWSKeyValueStore(Context context, String sharedPreferencesName, boolean isPersistenceEnabled) {
        this.cache = AWSKeyValueStore.getCacheForKey(sharedPreferencesName);
        this.sharedPreferencesName = sharedPreferencesName;
        this.context = context;
        this.setPersistenceEnabled(isPersistenceEnabled);
    }

    public synchronized void setPersistenceEnabled(boolean isPersistenceEnabled) {
        try {
            boolean previousIsPersistenceEnabled = this.isPersistenceEnabled;
            this.isPersistenceEnabled = isPersistenceEnabled;
            if (isPersistenceEnabled && !previousIsPersistenceEnabled) {
                this.sharedPreferencesForData = this.context.getSharedPreferences(this.sharedPreferencesName, 0);
                this.sharedPreferencesForEncryptionMaterials = this.context.getSharedPreferences(this.sharedPreferencesName + SHARED_PREFERENCES_ENCRYPTION_KEY_NAMESPACE_SUFFIX, 0);
                this.initKeyProviderBasedOnAPILevel();
                logger.info("Detected Android API Level = " + Build.VERSION.SDK_INT);
                logger.info("Creating the AWSKeyValueStore with key for sharedPreferencesForData = " + this.sharedPreferencesName);
                this.onMigrateFromNoEncryption();
            } else if (!isPersistenceEnabled) {
                logger.info("Persistence is disabled. Data will be accessed from memory.");
            }
            if (!isPersistenceEnabled && previousIsPersistenceEnabled) {
                this.sharedPreferencesForData.edit().clear().apply();
            }
        }
        catch (Exception ex) {
            logger.error("Error in enabling persistence for " + this.sharedPreferencesName, ex);
        }
    }

    public synchronized boolean contains(String dataKey) {
        if (this.isPersistenceEnabled) {
            if (this.cache.containsKey(dataKey)) {
                return true;
            }
            return this.sharedPreferencesForData.contains(this.getDataKeyUsedInPersistentStore(dataKey));
        }
        return this.cache.containsKey(dataKey);
    }

    public synchronized String get(String dataKey) {
        if (dataKey == null) {
            return null;
        }
        if (this.cache.containsKey(dataKey) || !this.isPersistenceEnabled) {
            return this.cache.get(dataKey);
        }
        String dataKeyInPersistentStore = this.getDataKeyUsedInPersistentStore(dataKey);
        String encryptionKeyAlias = this.getEncryptionKeyAlias();
        Key decryptionKey = this.retrieveEncryptionKey(encryptionKeyAlias);
        if (decryptionKey == null) {
            logger.warn("Error in retrieving the decryption key used to decrypt the data from the persistent store. Returning null for the requested dataKey = " + dataKey);
            return null;
        }
        if (!this.sharedPreferencesForData.contains(dataKeyInPersistentStore)) {
            return null;
        }
        try {
            int keyValueStoreVersion = Integer.parseInt(this.sharedPreferencesForData.getString(dataKeyInPersistentStore + SHARED_PREFERENCES_STORE_VERSION_SUFFIX, null));
            if (keyValueStoreVersion != 1) {
                logger.error("The version of the data read from SharedPreferences for " + dataKey + " does not match the version of the store.");
                return null;
            }
            String encryptedData = this.sharedPreferencesForData.getString(dataKeyInPersistentStore, null);
            String decryptedDataInString = this.decrypt(decryptionKey, this.getInitializationVector(dataKeyInPersistentStore), encryptedData);
            this.cache.put(dataKey, decryptedDataInString);
            return decryptedDataInString;
        }
        catch (Exception ex) {
            logger.warn("Error in retrieving value for dataKey = " + dataKey, ex);
            this.remove(dataKey);
            return null;
        }
    }

    public synchronized void put(String dataKey, String value) {
        if (dataKey == null) {
            logger.error("dataKey is null.");
            return;
        }
        this.cache.put(dataKey, value);
        if (!this.isPersistenceEnabled) {
            return;
        }
        if (value == null) {
            logger.debug("Value is null. Removing the data, IV and version from SharedPreferences");
            this.cache.remove(dataKey);
            this.remove(dataKey);
            return;
        }
        String dataKeyInPersistentStore = this.getDataKeyUsedInPersistentStore(dataKey);
        String encryptionKeyAlias = this.getEncryptionKeyAlias();
        Key encryptionKey = this.retrieveEncryptionKey(encryptionKeyAlias);
        if (encryptionKey == null) {
            logger.warn("No encryption key found for encryptionKeyAlias: " + encryptionKeyAlias);
            encryptionKey = this.generateEncryptionKey(encryptionKeyAlias);
            if (encryptionKey == null) {
                logger.warn("Error in generating the encryption key for encryptionKeyAlias: " + encryptionKeyAlias + " used to encrypt the data before storing. Skipping persisting the data in the persistent store.");
                return;
            }
        }
        try {
            byte[] iv = this.generateInitializationVector();
            if (iv == null) {
                throw new Exception("The generated IV for dataKey = " + dataKey + " is null.");
            }
            String base64EncodedEncryptedString = this.encrypt(encryptionKey, this.getAlgorithmParameterSpecForIV(iv), value);
            String base64EncodedIV = Base64.encodeAsString(iv);
            if (base64EncodedIV == null) {
                throw new Exception("Error in Base64 encoding the IV for dataKey = " + dataKey);
            }
            this.sharedPreferencesForData.edit().putString(dataKeyInPersistentStore, base64EncodedEncryptedString).putString(dataKeyInPersistentStore + SHARED_PREFERENCES_IV_SUFFIX, base64EncodedIV).putString(dataKeyInPersistentStore + SHARED_PREFERENCES_STORE_VERSION_SUFFIX, String.valueOf(1)).apply();
        }
        catch (Exception ex) {
            logger.error("Error in storing value for dataKey = " + dataKey + ". This data has not been stored in the persistent store.", ex);
        }
    }

    public synchronized void remove(String dataKey) {
        this.cache.remove(dataKey);
        if (this.isPersistenceEnabled) {
            String keyUsedInPersistentStore = this.getDataKeyUsedInPersistentStore(dataKey);
            this.sharedPreferencesForData.edit().remove(keyUsedInPersistentStore).remove(keyUsedInPersistentStore + SHARED_PREFERENCES_IV_SUFFIX).remove(keyUsedInPersistentStore + SHARED_PREFERENCES_STORE_VERSION_SUFFIX).apply();
        }
    }

    public synchronized void clear() {
        this.cache.clear();
        if (this.isPersistenceEnabled) {
            this.sharedPreferencesForData.edit().clear().apply();
        }
    }

    private String encrypt(Key encryptionKey, AlgorithmParameterSpec ivSpec, String data) {
        try {
            Cipher cipher = Cipher.getInstance(CIPHER_AES_GCM_NOPADDING);
            cipher.init(1, encryptionKey, ivSpec);
            byte[] encryptedData = cipher.doFinal(data.getBytes(CHARSET_NAME));
            return Base64.encodeAsString(encryptedData);
        }
        catch (Exception ex) {
            logger.error("Error in encrypting data. ", ex);
            return null;
        }
    }

    private String decrypt(Key decryptionKey, AlgorithmParameterSpec ivSpec, String encryptedData) {
        try {
            byte[] encryptedDecodedData = Base64.decode(encryptedData);
            Cipher cipher = Cipher.getInstance(CIPHER_AES_GCM_NOPADDING);
            cipher.init(2, decryptionKey, ivSpec);
            byte[] decryptedData = cipher.doFinal(encryptedDecodedData);
            return new String(decryptedData, CHARSET_NAME);
        }
        catch (Exception ex) {
            logger.error("Error in decrypting data. ", ex);
            return null;
        }
    }

    private AlgorithmParameterSpec getInitializationVector(String keyOfDataInSharedPreferences) throws Exception {
        String keyOfIV = keyOfDataInSharedPreferences + SHARED_PREFERENCES_IV_SUFFIX;
        if (!this.sharedPreferencesForData.contains(keyOfIV)) {
            throw new Exception("Initialization vector for " + keyOfDataInSharedPreferences + " is missing from the SharedPreferences.");
        }
        String initializationVectorInString = this.sharedPreferencesForData.getString(keyOfIV, null);
        if (initializationVectorInString == null) {
            throw new Exception("Cannot read the initialization vector for " + keyOfDataInSharedPreferences + " from SharedPreferences.");
        }
        byte[] base64DecodedIV = Base64.decode(initializationVectorInString);
        if (base64DecodedIV == null || base64DecodedIV.length == 0) {
            throw new Exception("Cannot base64 decode the initialization vector for " + keyOfDataInSharedPreferences + " read from SharedPreferences.");
        }
        return this.getAlgorithmParameterSpecForIV(base64DecodedIV);
    }

    private byte[] generateInitializationVector() {
        byte[] initializationVector = new byte[12];
        this.secureRandom.nextBytes(initializationVector);
        return initializationVector;
    }

    private AlgorithmParameterSpec getAlgorithmParameterSpecForIV(byte[] iv) {
        if (Build.VERSION.SDK_INT >= 23) {
            return new GCMParameterSpec(128, iv);
        }
        return new IvParameterSpec(iv);
    }

    private synchronized Key retrieveEncryptionKey(String encryptionKeyAlias) {
        try {
            return this.keyProvider.retrieveKey(encryptionKeyAlias);
        }
        catch (KeyNotFoundException keyNotFoundException) {
            logger.warn(keyNotFoundException);
            logger.info("Deleting the encryption key identified by the keyAlias: " + encryptionKeyAlias);
            this.keyProvider.deleteKey(encryptionKeyAlias);
            return null;
        }
    }

    synchronized Key generateEncryptionKey(String encryptionKeyAlias) {
        try {
            return this.keyProvider.generateKey(encryptionKeyAlias);
        }
        catch (KeyNotGeneratedException keyNotGeneratedException) {
            logger.error("Encryption Key cannot be generated successfully.", keyNotGeneratedException);
            return null;
        }
    }

    private String getDataKeyUsedInPersistentStore(String key) {
        if (key == null) {
            return null;
        }
        return key + SHARED_PREFERENCES_DATA_IDENTIFIER_SUFFIX;
    }

    private String getEncryptionKeyAlias() {
        if (Build.VERSION.SDK_INT >= 23) {
            return this.sharedPreferencesName + ".aesKeyStoreAlias";
        }
        if (Build.VERSION.SDK_INT >= 18) {
            return this.sharedPreferencesName + ".rsaKeyStoreAlias";
        }
        if (Build.VERSION.SDK_INT >= 10) {
            return "AesGcmNoPaddingEncryption10-encryption-key";
        }
        logger.error("API Level " + String.valueOf(Build.VERSION.SDK_INT) + " not supported by the SDK. Setting persistence to false.");
        this.isPersistenceEnabled = false;
        return null;
    }

    private void initKeyProviderBasedOnAPILevel() {
        if (Build.VERSION.SDK_INT >= 23) {
            this.keyProvider = new KeyProvider23();
        } else if (Build.VERSION.SDK_INT >= 18) {
            this.keyProvider = new KeyProvider18(this.context, this.sharedPreferencesForEncryptionMaterials);
        } else if (Build.VERSION.SDK_INT >= 10) {
            this.keyProvider = new KeyProvider10(this.sharedPreferencesForEncryptionMaterials);
        } else {
            logger.error("API Level " + String.valueOf(Build.VERSION.SDK_INT) + " not supported by the SDK. Setting persistence to false.");
            this.isPersistenceEnabled = false;
        }
    }

    private void onMigrateFromNoEncryption() {
        Map map = this.sharedPreferencesForData.getAll();
        for (String keyOfUnencryptedData : map.keySet()) {
            if (keyOfUnencryptedData.endsWith(SHARED_PREFERENCES_DATA_IDENTIFIER_SUFFIX) || keyOfUnencryptedData.endsWith(SHARED_PREFERENCES_IV_SUFFIX) || keyOfUnencryptedData.endsWith(SHARED_PREFERENCES_STORE_VERSION_SUFFIX)) continue;
            if (map.get(keyOfUnencryptedData) instanceof Long) {
                Long longValue = this.sharedPreferencesForData.getLong(keyOfUnencryptedData, 0L);
                this.put(keyOfUnencryptedData, String.valueOf(longValue));
            } else if (map.get(keyOfUnencryptedData) instanceof String) {
                this.put(keyOfUnencryptedData, this.sharedPreferencesForData.getString(keyOfUnencryptedData, null));
            } else if (map.get(keyOfUnencryptedData) instanceof Float) {
                Float floatValue = Float.valueOf(this.sharedPreferencesForData.getFloat(keyOfUnencryptedData, 0.0f));
                this.put(keyOfUnencryptedData, String.valueOf(floatValue));
            } else if (map.get(keyOfUnencryptedData) instanceof Boolean) {
                Boolean booleanValue = this.sharedPreferencesForData.getBoolean(keyOfUnencryptedData, false);
                this.put(keyOfUnencryptedData, String.valueOf(booleanValue));
            } else if (map.get(keyOfUnencryptedData) instanceof Integer) {
                Integer intValue = this.sharedPreferencesForData.getInt(keyOfUnencryptedData, 0);
                this.put(keyOfUnencryptedData, String.valueOf(intValue));
            } else if (map.get(keyOfUnencryptedData) instanceof Set) {
                Set stringSet = (Set)map.get(keyOfUnencryptedData);
                StringBuilder stringBuilder = new StringBuilder();
                Iterator setIterator = stringSet.iterator();
                while (setIterator.hasNext()) {
                    stringBuilder.append((String)setIterator.next());
                    if (!setIterator.hasNext()) continue;
                    stringBuilder.append(",");
                }
                this.put(keyOfUnencryptedData, stringBuilder.toString());
            }
            this.sharedPreferencesForData.edit().remove(keyOfUnencryptedData).apply();
        }
    }
}

