/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.services.kinesis.leases.impl;

import com.amazonaws.services.cloudwatch.model.StandardUnit;
import com.amazonaws.services.kinesis.leases.exceptions.DependencyException;
import com.amazonaws.services.kinesis.leases.exceptions.InvalidStateException;
import com.amazonaws.services.kinesis.leases.exceptions.ProvisionedThroughputException;
import com.amazonaws.services.kinesis.leases.impl.Lease;
import com.amazonaws.services.kinesis.leases.interfaces.ILeaseManager;
import com.amazonaws.services.kinesis.leases.interfaces.ILeaseRenewer;
import com.amazonaws.services.kinesis.metrics.impl.MetricsHelper;
import com.amazonaws.services.kinesis.metrics.interfaces.MetricsLevel;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class LeaseRenewer<T extends Lease>
implements ILeaseRenewer<T> {
    private static final Log LOG = LogFactory.getLog(LeaseRenewer.class);
    private static final int RENEWAL_RETRIES = 2;
    private final ILeaseManager<T> leaseManager;
    private final ConcurrentNavigableMap<String, T> ownedLeases = new ConcurrentSkipListMap<String, T>();
    private final String workerIdentifier;
    private final long leaseDurationNanos;

    public LeaseRenewer(ILeaseManager<T> leaseManager, String workerIdentifier, long leaseDurationMillis) {
        this.leaseManager = leaseManager;
        this.workerIdentifier = workerIdentifier;
        this.leaseDurationNanos = TimeUnit.MILLISECONDS.toNanos(leaseDurationMillis);
    }

    @Override
    public void renewLeases() throws DependencyException, InvalidStateException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)String.format("Worker %s holding %d leases: %s", this.workerIdentifier, this.ownedLeases.size(), this.ownedLeases));
        }
        int lostLeases = 0;
        for (Lease lease : this.ownedLeases.descendingMap().values()) {
            if (this.renewLease(lease)) continue;
            ++lostLeases;
        }
        MetricsHelper.getMetricsScope().addData("LostLeases", lostLeases, StandardUnit.Count, MetricsLevel.SUMMARY);
        MetricsHelper.getMetricsScope().addData("CurrentLeases", this.ownedLeases.size(), StandardUnit.Count, MetricsLevel.SUMMARY);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean renewLease(T lease) throws DependencyException, InvalidStateException {
        String leaseKey = ((Lease)lease).getLeaseKey();
        boolean success = false;
        boolean renewedLease = false;
        long startTime = System.currentTimeMillis();
        try {
            for (int i = 1; i <= 2; ++i) {
                try {
                    T t = lease;
                    synchronized (t) {
                        renewedLease = this.leaseManager.renewLease(lease);
                        if (renewedLease) {
                            ((Lease)lease).setLastCounterIncrementNanos(System.nanoTime());
                        }
                    }
                    if (renewedLease) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug((Object)String.format("Worker %s successfully renewed lease with key %s", this.workerIdentifier, leaseKey));
                        }
                    } else {
                        LOG.info((Object)String.format("Worker %s lost lease with key %s", this.workerIdentifier, leaseKey));
                        this.ownedLeases.remove(leaseKey);
                    }
                    success = true;
                    break;
                }
                catch (ProvisionedThroughputException e) {
                    LOG.info((Object)String.format("Worker %s could not renew lease with key %s on try %d out of %d due to capacity", this.workerIdentifier, leaseKey, i, 2));
                    continue;
                }
            }
        }
        finally {
            MetricsHelper.addSuccessAndLatency("RenewLease", startTime, success, MetricsLevel.DETAILED);
        }
        return renewedLease;
    }

    @Override
    public Map<String, T> getCurrentlyHeldLeases() {
        HashMap<String, T> result = new HashMap<String, T>();
        long now = System.nanoTime();
        for (String leaseKey : this.ownedLeases.keySet()) {
            T copy = this.getCopyOfHeldLease(leaseKey, now);
            if (copy == null) continue;
            result.put(((Lease)copy).getLeaseKey(), copy);
        }
        return result;
    }

    @Override
    public T getCurrentlyHeldLease(String leaseKey) {
        return this.getCopyOfHeldLease(leaseKey, System.nanoTime());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private T getCopyOfHeldLease(String leaseKey, long now) {
        Lease authoritativeLease = (Lease)this.ownedLeases.get(leaseKey);
        if (authoritativeLease == null) {
            return null;
        }
        Lease copy = null;
        Lease lease = authoritativeLease;
        synchronized (lease) {
            copy = (Lease)authoritativeLease.copy();
        }
        if (copy.isExpired(this.leaseDurationNanos, now)) {
            LOG.info((Object)String.format("getCurrentlyHeldLease not returning lease with key %s because it is expired", copy.getLeaseKey()));
            return null;
        }
        return (T)copy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean updateLease(T lease, UUID concurrencyToken) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        this.verifyNotNull(lease, "lease cannot be null");
        this.verifyNotNull(((Lease)lease).getLeaseKey(), "leaseKey cannot be null");
        this.verifyNotNull(concurrencyToken, "concurrencyToken cannot be null");
        String leaseKey = ((Lease)lease).getLeaseKey();
        Lease authoritativeLease = (Lease)this.ownedLeases.get(leaseKey);
        if (authoritativeLease == null) {
            LOG.info((Object)String.format("Worker %s could not update lease with key %s because it does not hold it", this.workerIdentifier, leaseKey));
            return false;
        }
        if (!authoritativeLease.getConcurrencyToken().equals(concurrencyToken)) {
            LOG.info((Object)String.format("Worker %s refusing to update lease with key %s because concurrency tokens don't match", this.workerIdentifier, leaseKey));
            return false;
        }
        long startTime = System.currentTimeMillis();
        boolean success = false;
        try {
            Lease lease2 = authoritativeLease;
            synchronized (lease2) {
                authoritativeLease.update(lease);
                boolean updatedLease = this.leaseManager.updateLease(authoritativeLease);
                if (updatedLease) {
                    authoritativeLease.setLastCounterIncrementNanos(System.nanoTime());
                } else {
                    LOG.info((Object)String.format("Worker %s lost lease with key %s - discovered during update", this.workerIdentifier, leaseKey));
                    this.ownedLeases.remove(leaseKey, authoritativeLease);
                }
                success = true;
                boolean bl = updatedLease;
                return bl;
            }
        }
        finally {
            MetricsHelper.addSuccessAndLatency("UpdateLease", startTime, success, MetricsLevel.DETAILED);
        }
    }

    @Override
    public void addLeasesToRenew(Collection<T> newLeases) {
        this.verifyNotNull(newLeases, "newLeases cannot be null");
        for (Lease lease : newLeases) {
            if (lease.getLastCounterIncrementNanos() == null) {
                LOG.info((Object)String.format("addLeasesToRenew ignoring lease with key %s because it does not have lastRenewalNanos set", lease.getLeaseKey()));
                continue;
            }
            Object authoritativeLease = lease.copy();
            ((Lease)authoritativeLease).setConcurrencyToken(UUID.randomUUID());
            this.ownedLeases.put(((Lease)authoritativeLease).getLeaseKey(), authoritativeLease);
        }
    }

    @Override
    public void clearCurrentlyHeldLeases() {
        this.ownedLeases.clear();
    }

    @Override
    public void initialize() throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        List<T> leases = this.leaseManager.listLeases();
        LinkedList<Lease> myLeases = new LinkedList<Lease>();
        for (Lease lease : leases) {
            if (this.workerIdentifier.equals(lease.getLeaseOwner())) {
                LOG.info((Object)String.format(" Worker %s found lease %s", this.workerIdentifier, lease));
                if (!this.renewLease(lease)) continue;
                myLeases.add(lease);
                continue;
            }
            LOG.debug((Object)String.format("Worker %s ignoring lease %s ", this.workerIdentifier, lease));
        }
        this.addLeasesToRenew(myLeases);
    }

    private void verifyNotNull(Object object, String message) {
        if (object == null) {
            throw new IllegalArgumentException(message);
        }
    }
}

