/**
 * Copyright 2016-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package com.amazon.pay.request;

import com.amazon.pay.types.CurrencyCode;
import com.amazon.pay.response.model.ProviderCredit;
import java.io.Serializable;
import java.util.List;

/**
 * Container for the parameters to the Refund operation}.
 * For more information documentation, see
 * https://pay.amazon.com/documentation/
 */
public class RefundRequest extends DelegateRequest<RefundRequest> implements Serializable {

    @Override
    protected RefundRequest getThis() {
        return this;
    }

    //required parameters
    private String amazonCaptureId;
    private String refundReferenceId;
    private String refundAmount;
    private CurrencyCode refundCurrencyCode;

    //optional parameters
    private String sellerRefundNote;
    private String softDescriptor;
    private List<ProviderCredit> providerCredit;

    /**
     * @param captureId The capture identifier that was generated by Amazon in the earlier call to Capture.
     * @param refundReferenceId The identifier for this refund transaction that you specify.
     *                          This identifier must be unique for all your refund transactions.
     * @param refundAmount The amount to refund.
     */
    public RefundRequest(String captureId, String refundReferenceId, String refundAmount ) {
        this.amazonCaptureId = captureId;
        this.refundReferenceId = refundReferenceId;
        this.refundAmount = refundAmount;
    }

    /**
     * @param currencyCode Three-digit currency code. In ISO 4217 format.
     *
     * @return the Currency Code
     */
    public RefundRequest setRefundCurrencyCode(CurrencyCode currencyCode) {
        this.refundCurrencyCode = currencyCode;
        return this;
    }


    /**
     * @param sellerRefundNote Sets the description for the refund that is
     *                         displayed in emails to the buyer.
     *
     * @return the Seller Refund Note
     */
    public RefundRequest setSellerRefundNote(String sellerRefundNote){
        this.sellerRefundNote = sellerRefundNote;
        return this;
    }

    /**
     * @param softDescriptor Sets the description to be shown on the buyer's
     *                       payment instrument statement.
     *
     * @return the Soft Descriptor
     */
    public RefundRequest setSoftDescriptor(String softDescriptor) {
        this.softDescriptor = softDescriptor;
        return this;
    }

    /**
     * @return amazonCaptureId The capture identifier that was generated by
     *         Amazon in the earlier call to Capture.
     */
    public String getAmazonCaptureId() {
        return amazonCaptureId;
    }

    /**
     * @return refundReferenceId The identifier for this refund transaction
     *         that you specify.
     */
    public String getRefundReferenceId() {
        return refundReferenceId;
    }

    /**
     * @return refundAmount The amount to refund.
     */
    public String getRefundAmount() {
        return refundAmount;
    }

    /**
     * @return refundCurrencyCode Three-digit currency code.
     */
    public CurrencyCode getRefundCurrencyCode() {
        return refundCurrencyCode;
    }

    /**
     * @return sellerRefundNote The description for the refund that is
     *         displayed in emails to the buyer.
     */
    public String getSellerRefundNote() {
        return sellerRefundNote;
    }

    /**
     * @return softDescriptor The description to be shown on the buyer's
     *         payment instrument statement.
     */
    public String getSoftDescriptor() {
        return softDescriptor;
    }

    /**
     * @return ProviderCredit associated with Refund operation
     */
    public List<ProviderCredit> getProviderCredit() {
        return providerCredit;
    }

    /**
     * @param providerCreditList associated with Refund operation
     *
     * @return Provider Credit list
     *
     */
    public RefundRequest setProviderCredit(List<ProviderCredit> providerCreditList) {
        this.providerCredit = providerCreditList;
        return this;
    }

    /**
     * Returns a string representation of this object; useful for testing and
     * debugging.
     *
     * @return A string representation of this object.
     *
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "RefundRequest{"
                + "amazonCaptureId=" + amazonCaptureId
                + ", refundReferenceId=" + refundReferenceId
                + ", refundAmount=" + refundAmount
                + ", refundCurrencyCode=" + refundCurrencyCode
                + ", sellerRefundNote=" + sellerRefundNote
                + ", softDescriptor=" + softDescriptor
                + ", mwsAuthToken=" + getMwsAuthToken()
                + ", providerCredit=" + providerCredit + '}';
    }

}
