/**
 * Copyright 2016-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package com.amazon.pay.request;

import com.amazon.pay.types.CurrencyCode;
import com.amazon.pay.response.model.ProviderCredit;
import java.io.Serializable;
import java.util.List;

/**
 *  For more information documentation, see
 * https://pay.amazon.com/documentation/
 */

public class CaptureRequest extends DelegateRequest<CaptureRequest> implements Serializable {

    @Override
    protected CaptureRequest getThis() {
        return this;
    }

    //required parameters
    private String amazonAuthorizationId;
    private String captureReferenceId;
    private String captureAmount;
    private CurrencyCode captureCurrencyCode;

    //optional parameters
    private String sellerCaptureNote;
    private String softDescriptor;
    private List<ProviderCredit> providerCredit;

    /*
    * @param amazonAuthorizationId
    *               The authorization identifier that was generated by
    *               Amazon in the earlier call to Authorize or AuthorizeOnBillingAgreement.
    * @param captureReferenceId
    *               The identifier for this capture transaction that you specify.
    *               This identifier must be unique for all your capture transactions.
    * @param captureAmount
    *               The amount to capture in this transaction.
    * @return
    *           Container holding Capture operation parameters
    *
    */
    public CaptureRequest(String amazonAuthorizationId, String captureReferenceId, String captureAmount) {
        this.amazonAuthorizationId = amazonAuthorizationId;
        this.captureReferenceId = captureReferenceId;
        this.captureAmount = captureAmount;

    }
    /**
     * @param currencyCode
     *       Three-digit currency code. In ISO 4217 format.
     *
     * @return Returns a reference to this object so that methods can be chained together.
     */
    public CaptureRequest setCaptureCurrencyCode(CurrencyCode currencyCode) {
        this.captureCurrencyCode = currencyCode;
        return this;
    }

    /**
     * @param sellerCaptureNote
     *       A description for the capture transaction that is displayed in emails to the buyer.
     *
     * @return Returns a reference to this object so that methods can be chained together.
     */
    public CaptureRequest setSellerCaptureNote(String sellerCaptureNote) {
        this.sellerCaptureNote = sellerCaptureNote;
        return this;
    }

    /**
     * @param softDescriptor The description to be shown on the buyer's payment instrument statement.
     *                       The soft descriptor sent to the payment processor is:
     *                       “AMZ* &lt;soft descriptor specified here&gt;”.
     *
     * @return Returns a reference to this object so that methods can be chained together.
     */
    public CaptureRequest setSoftDescriptor(String softDescriptor) {
        this.softDescriptor = softDescriptor;
        return this;
    }

    /**
     * Applicable for third-party solution providers only.
     *
     * @param providerCreditList associated with Authorize operation
     *
     * @return Returns a reference to this object so that methods can be chained together.
     */
    public CaptureRequest setProviderCredit(List<ProviderCredit> providerCreditList) {
        this.providerCredit = providerCreditList;
        return this;
    }

    /**
     * The authorization identifier that was generated by
     * Amazon in the earlier call to Authorize or AuthorizeOnBillingAgreement.
     *
     * @return Authorization identifier specified in request
     */
    public String getAmazonAuthorizationId() {
        return amazonAuthorizationId;
    }

    /**
     * The identifier for this capture transaction that you specify. 
     *
     * @return Capture Reference Id specified in request
     */
    public String getCaptureReferenceId() {
        return captureReferenceId;
    }

    /**
     *  The amount to capture in this transaction.
     *
     * @return Capture amount specified in request
     */
    public String getCaptureAmount() {
        return captureAmount;
    }

    /**
     * Three-digit currency code. In ISO 4217 format.
     *
     * @return Currency code specified in request
     */
    public CurrencyCode getCaptureCurrencyCode() {
        return captureCurrencyCode;
    }

    /**
     * A description for the capture transaction that is displayed in emails to the buyer.
     *
     * @return SellerCaptureNote specified in request
     */
    public String getSellerCaptureNote() {
        return sellerCaptureNote;
    }

    /**
     * The description to be shown on the buyer's payment instrument statement. 
     *                       The soft descriptor sent to the payment processor is:
     *                       “AMZ* &lt;soft descriptor specified here&gt;”.
     *
     * @return Soft descriptor string for Capture operation
     */
    public String getSoftDescriptor() {
        return softDescriptor;
    }

    /**
     * Applicable for third-party solution providers only.
     *
     * @return ProviderCredit associated with Authorize operation
     */
    public List<ProviderCredit> getProviderCredit() {
        return providerCredit;
    }

    /**
     * Returns a string representation of this object; useful for testing and
     * debugging.
     *
     * @return A string representation of this object.
     *
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "CaptureRequest{"
                + "amazonAuthorizationId=" + amazonAuthorizationId
                + ", captureReferenceId=" + captureReferenceId
                + ", captureAmount=" + captureAmount
                + ", captureCurrencyCode=" + captureCurrencyCode
                + ", sellerCaptureNote=" + sellerCaptureNote
                + ", softDescriptor=" + softDescriptor
                + ", mwsAuthToken=" + getMwsAuthToken()
                + ", providerCredit=" + providerCredit + '}';
    }

}
