/*
 * Copyright (C) Jerry Huxtable 1998
 */
package com.alkacon.simapi.filter.math;

import com.alkacon.simapi.filter.*;

import java.awt.*;
import java.awt.image.*;

public class ImageFunction2D implements Function2D {

	public final static int ZERO = 0;
	public final static int CLAMP = 1;
	public final static int WRAP = 2;
	
	protected int[] pixels;
	protected int width;
	protected int height;
	protected int edgeAction = ZERO;
	
	public ImageFunction2D(Image image) {
		this(image, ZERO);
	}
	
	public ImageFunction2D(Image image, int edgeAction) {
		PixelGrabber pg = new PixelGrabber(image, 0, 0, -1, -1, null, 0, -1);
		try {
			pg.grabPixels();
		} catch (InterruptedException e) {
			throw new RuntimeException("interrupted waiting for pixels!");
		}
		if ((pg.status() & ImageObserver.ABORT) != 0) {
			throw new RuntimeException("image fetch aborted");
		}
		init((int[])pg.getPixels(), pg.getWidth(), pg.getHeight(), edgeAction);
	}
	
	public ImageFunction2D(int[] pixels, int width, int height, int edgeAction) {
		init(pixels, width, height, edgeAction);
	}
	
	public void init(int[] pixels, int width, int height, int edgeAction) {
		this.pixels = pixels;
		this.width = width;
		this.height = height;
		this.edgeAction = edgeAction;
	}
	
	public float evaluate(float x, float y) {
		int ix = (int)x;
		int iy = (int)y;
		if (edgeAction == WRAP) {
			ix = ImageMath.mod(ix, width);
			iy = ImageMath.mod(iy, height);
		} else if (ix < 0 || iy < 0 || ix >= width || iy >= height) {
			if (edgeAction == ZERO)
				return 0;
			if (ix < 0)
				ix = 0;
			else if (ix >= width)
				ix = width-1;
			if (iy < 0)
				iy = 0;
			else if (iy >= height)
				iy = height-1;
		}
		return PixelUtils.brightness(pixels[iy*width+ix]) / 255.0f;
	}
	
	public void setEdgeAction(int edgeAction) {
		this.edgeAction = edgeAction;
	}

	public int getEdgeAction() {
		return edgeAction;
	}

	public int getWidth() {
		return width;
	}
	
	public int getHeight() {
		return height;
	}
	
}

