/*
 * Copyright (C) Jerry Huxtable 1998
 */
package com.alkacon.simapi.filter;

import java.awt.image.*;

public class MotionBlurFilter extends WholeImageFilter {

	public final static int LINEAR = 0;
	public final static int RADIAL = 1;
	public final static int ZOOM = 2;
	
	private float angle = 0.0f;
	private float falloff = 1.0f;
	private float distance = 1.0f;
	private int repetitions = 4;
	private int type = LINEAR;

	public MotionBlurFilter() {
	}

	public void setType(int type) {
		this.type = type;
	}

	public int getType() {
		return type;
	}

	public void setAngle(float angle) {
		this.angle = angle;
	}

	public float getAngle() {
		return angle;
	}

	public void setDistance(float distance) {
		this.distance = distance;
	}

	public float getDistance() {
		return distance;
	}

	public void setRepetitions(int repetitions) {
		this.repetitions = repetitions;
	}

	public int getRepetitions() {
		return repetitions;
	}

	public void imageComplete(int status) {
		if (status == IMAGEERROR || status == IMAGEABORTED) {
			consumer.imageComplete(status);
			return;
		}

		int width = originalSpace.width;
		int height = originalSpace.height;
		int index = 0;
		int[] outPixels = new int[width * height];

		float sinAngle = (float)Math.sin(angle);
		float cosAngle = (float)Math.cos(angle);

		float total;
		int cx = width/2;
		int cy = height/2;

		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				int a = 0, r = 0, g = 0, b = 0;
				int count = 0;
				for (int i = 0; i < repetitions; i++) {
					int newX = x, newY = y;
					if (i != 0) {
						switch (type) {
						case LINEAR:
						default:
							newX = (int)(x + i*distance*sinAngle);
							newY = (int)(y + i*distance*cosAngle);
							break;
						case RADIAL:
							{
							float dx = x-cx;
							float dy = y-cy;
							float d = (float)Math.sqrt(dx*dx+dy*dy);
							float an = (float)Math.atan2(dy, dx);
							an += distance * (i-repetitions/2) / ImageMath.PI/10.0;
							newX = (int)(cx + d*Math.cos(an));
							newY = (int)(cy + d*Math.sin(an));
							}
							break;
						case ZOOM:
							{
							float dx = x-cx;
							float dy = y-cy;
							newX = (int)(cx + (1.0-i*distance/100.0)*dx);
							newY = (int)(cy + (1.0-i*distance/100.0)*dy);
							}
							break;
						}
						if (newX < 0)
							break;
						else if (newX >= width)
							break;
						if (newY < 0)
							break;
						else if (newY >= height)
							break;
					}
					count++;
					int rgb = inPixels[newY*width+newX];
					a += (rgb >> 24) & 0xff;
					r += (rgb >> 16) & 0xff;
					g += (rgb >> 8) & 0xff;
					b += rgb & 0xff;
				}
				if (count == 0) {
					outPixels[index] = inPixels[index];
				} else {
					a = PixelUtils.clamp((int)(a/count));
					r = PixelUtils.clamp((int)(r/count));
					g = PixelUtils.clamp((int)(g/count));
					b = PixelUtils.clamp((int)(b/count));
					outPixels[index] = (a << 24) | (r << 16) | (g << 8) | b;
				}
				index++;
			}
		}
		consumer.setPixels(0, 0, width, height, defaultRGBModel, outPixels, 0, width);
		consumer.imageComplete(status);
		inPixels = null;
	}

	public String toString() {
		return "Blur/Motion Blur...";
	}
}

