/*
 * Decompiled with CFR 0.152.
 */
package com.alibaba.cloud.ai.graph.agent.extension.tools.filesystem;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.function.BiFunction;
import org.springframework.ai.chat.model.ToolContext;
import org.springframework.ai.tool.ToolCallback;
import org.springframework.ai.tool.function.FunctionToolCallback;

public class ReadFileTool
implements BiFunction<ReadFileRequest, ToolContext, String> {
    private static final String EMPTY_CONTENT_WARNING = "System reminder: File exists but has empty contents";
    private static final int MAX_LINE_LENGTH = 10000;
    private static final int LINE_NUMBER_WIDTH = 6;
    public static final String DESCRIPTION = "Reads a file from the filesystem. You can access any file directly by using this tool.\nAssume this tool is able to read all files on the machine. If the User provides a path to a file assume that path is valid. It is okay to read a file that does not exist; an error will be returned.\n\nUsage:\n- The file_path parameter must be an absolute path, not a relative path\n- By default, it reads up to 500 lines starting from the beginning of the file\n- **IMPORTANT for large files and codebase exploration**: Use pagination with offset and limit parameters to avoid context overflow\n  - First scan: read_file(path, limit=100) to see file structure\n  - Read more sections: read_file(path, offset=100, limit=200) for next 200 lines\n  - Only omit limit (read full file) when necessary for editing\n- Specify offset and limit: read_file(path, offset=0, limit=100) reads first 100 lines\n- Any lines longer than 2000 characters will be truncated\n- Results are returned using cat -n format, with line numbers starting at 1\n- You have the capability to call multiple tools in a single response. It is always better to speculatively read multiple files as a batch that are potentially useful.\n- If you read a file that exists but has empty contents you will receive a system reminder warning in place of file contents.\n\t\t\t- You should almost ALWAYS use the list_files tool before using this tool to verify the file path.\n";

    @Override
    public String apply(ReadFileRequest request, ToolContext toolContext) {
        try {
            Path path = Paths.get(request.filePath, new String[0]);
            return ReadFileTool.readFileContent(path, request.offset, request.limit, true);
        }
        catch (Exception e) {
            return "Error reading file: " + e.getMessage();
        }
    }

    public static String readFileContent(Path filePath, Integer offset, Integer limit, boolean checkEmpty) {
        try {
            String emptyMsg;
            if (!Files.exists(filePath, new LinkOption[0]) || !Files.isRegularFile(filePath, LinkOption.NOFOLLOW_LINKS)) {
                return "Error: File '" + filePath + "' not found";
            }
            String content = Files.readString(filePath);
            if (checkEmpty && (emptyMsg = ReadFileTool.checkEmptyContent(content)) != null) {
                return emptyMsg;
            }
            String[] lines = content.split("\n", -1);
            if (lines.length > 0 && lines[lines.length - 1].isEmpty()) {
                lines = Arrays.copyOf(lines, lines.length - 1);
            }
            int startIdx = offset != null ? offset : 0;
            int endIdx = Math.min(startIdx + (limit != null ? limit : 500), lines.length);
            if (startIdx >= lines.length) {
                return "Error: Line offset " + startIdx + " exceeds file length (" + lines.length + " lines)";
            }
            String[] selectedLines = Arrays.copyOfRange(lines, startIdx, endIdx);
            return ReadFileTool.formatContentWithLineNumbers(selectedLines, startIdx + 1);
        }
        catch (IOException e) {
            return "Error reading file '" + filePath + "': " + e.getMessage();
        }
    }

    private static String checkEmptyContent(String content) {
        if (content == null || content.trim().isEmpty()) {
            return EMPTY_CONTENT_WARNING;
        }
        return null;
    }

    private static String formatContentWithLineNumbers(String[] lines, int startLine) {
        StringBuilder result = new StringBuilder();
        for (int i = 0; i < lines.length; ++i) {
            String line = lines[i];
            int lineNum = i + startLine;
            if (line.length() <= 10000) {
                result.append(String.format("%6d\t%s\n", lineNum, line));
                continue;
            }
            int numChunks = (line.length() + 10000 - 1) / 10000;
            for (int chunkIdx = 0; chunkIdx < numChunks; ++chunkIdx) {
                int start = chunkIdx * 10000;
                int end = Math.min(start + 10000, line.length());
                String chunk = line.substring(start, end);
                if (chunkIdx == 0) {
                    result.append(String.format("%6d\t%s\n", lineNum, chunk));
                    continue;
                }
                String continuationMarker = lineNum + "." + chunkIdx;
                result.append(String.format("%6s\t%s\n", continuationMarker, chunk));
            }
        }
        if (!result.isEmpty() && result.charAt(result.length() - 1) == '\n') {
            result.setLength(result.length() - 1);
        }
        return result.toString();
    }

    public static ToolCallback createReadFileToolCallback(String description) {
        return FunctionToolCallback.builder((String)"read_file", (BiFunction)new ReadFileTool()).description(description).inputType(ReadFileRequest.class).build();
    }

    public static class ReadFileRequest {
        @JsonProperty(required=true, value="file_path")
        @JsonPropertyDescription(value="The absolute path of the file to read")
        public String filePath;
        @JsonProperty(value="offset")
        @JsonPropertyDescription(value="Line offset to start reading from (default: 0)")
        public Integer offset;
        @JsonProperty(value="limit")
        @JsonPropertyDescription(value="Maximum number of lines to read (default: 500)")
        public Integer limit;

        public ReadFileRequest() {
        }

        public ReadFileRequest(String filePath, Integer offset, Integer limit) {
            this.filePath = filePath;
            this.offset = offset;
            this.limit = limit;
        }
    }
}

