package com.algolia.instantsearch.insights

import android.content.Context
import com.algolia.instantsearch.insights.converter.ConverterEventToEventInternal
import com.algolia.instantsearch.insights.database.Database
import com.algolia.instantsearch.insights.database.DatabaseSharedPreferences
import com.algolia.instantsearch.insights.event.*
import com.algolia.instantsearch.insights.webservice.WebService
import com.algolia.instantsearch.insights.webservice.WebServiceHttp

/**
 * Main class used for interacting with the InstantSearch Insights library.
 * In order to send insights, you first need to register an APP ID and API key for a given Index.
 * Once registered, you can simply call `Insights.shared(index: String)` to send your events.
 */
class Insights internal constructor(
    private val indexName: String,
    private val eventUploader: EventUploader,
    internal val database: Database,
    internal val webService: WebService
) {

    /**
     * Insights configuration.
     * @param connectTimeoutInMilliseconds Maximum amount of time in milliseconds before a connect timeout.
     * @param readTimeoutInMilliseconds Maximum amount of time in milliseconds before a read timeout.
     */
    class Configuration @JvmOverloads constructor(
        val connectTimeoutInMilliseconds: Int,
        val readTimeoutInMilliseconds: Int,
        val defaultUserToken: String? = null
    )


    /**
     * Change this variable to `true` or `false` to enable or disable logging.
     * Use a filter on tag `Algolia Insights` to see all logs generated by the Insights library.
     */
    @Suppress("unused") // setter does side-effect
    var loggingEnabled: Boolean = false
        set(value) {
            field = value
            InsightsLogger.enabled[indexName] = value
        }

    /**
     * Change this variable to `true` or `false` to disable Insights, opting-out the current session from tracking.
     */
    var enabled: Boolean = true

    /**
     * Change this variable to change the default debouncing interval. Values lower than 15 minutes will be ignored.
     */
    var debouncingIntervalInMinutes: Long? = null
        set(value) {
            value?.let { eventUploader.setInterval(value) }
        }

    /**
     * Set a user identifier that will override any event's.
     *
     * Depending if the user is logged-in or not, several strategies can be used from a sessionId to a technical identifier.
     * You should always send pseudonymous or anonymous userTokens.
     */
    var userToken: String? = null

    private fun userTokenOrThrow(): String = userToken ?: throw InsightsException.NoUserToken()

    /**
     * Change this variable to change the default amount of event sent at once.
     */
    var minBatchSize: Int = 10

    init {
        eventUploader.startPeriodicUpload()
    }

    // region Event tracking methods
    /**
     * Tracks a View event, unrelated to a specific search query.
     *
     * @param eventName the event's name, **must not be empty**.
     * @param objectIDs the viewed object(s)' `objectID`.
     * @param timestamp the time at which the view happened. Defaults to current time.
     */
    @JvmOverloads
    fun viewed(
        eventName: String,
        objectIDs: EventObjects.IDs,
        timestamp: Long = System.currentTimeMillis()
    ) = viewed(
        Event.View(
            eventName = eventName,
            userToken = userTokenOrThrow(),
            timestamp = timestamp,
            eventObjects = objectIDs
        )
    )

    /**
     * Tracks a View event, unrelated to a specific search query.
     *
     * @param eventName the event's name, **must not be empty**.
     * @param filters the clicked filter(s).
     * @param timestamp the time at which the view happened. Defaults to current time.
     */
    @JvmOverloads
    fun viewed(
        eventName: String,
        filters: EventObjects.Filters,
        timestamp: Long = System.currentTimeMillis()
    ) = viewed(
        Event.View(
            eventName = eventName,
            userToken = userTokenOrThrow(),
            timestamp = timestamp,
            eventObjects = filters
        )
    )

    /**
     * Tracks a click event, unrelated to a specific search query.
     *
     * @param eventName the event's name, **must not be empty**.
     * @param objectIDs the clicked object(s)' `objectID`.
     * @param timestamp the time at which the click happened. Defaults to current time.
     */
    @JvmOverloads
    fun clicked(
        eventName: String,
        objectIDs: EventObjects.IDs,
        timestamp: Long = System.currentTimeMillis()
    ) = clicked(
        Event.Click(
            eventName = eventName,
            userToken = userTokenOrThrow(),
            timestamp = timestamp,
            eventObjects = objectIDs
        )
    )


    /**
     * Tracks a click event, unrelated to a specific search query.
     *
     * @param eventName the event's name, **must not be empty**.
     * @param filters the clicked filter(s).
     * @param timestamp the time at which the click happened. Defaults to current time.
     */
    @JvmOverloads
    fun clicked(
        eventName: String,
        filters: EventObjects.Filters,
        timestamp: Long = System.currentTimeMillis()
    ) = clicked(
        Event.Click(
            eventName = eventName,
            userToken = userTokenOrThrow(),
            timestamp = timestamp,
            eventObjects = filters
        )
    )

    /**
     * Tracks a Click event after a search has been done.
     *
     * @param eventName the event's name, **must not be empty**.
     * @param queryId the related [query's identifier][https://www.algolia.com/doc/guides/insights-and-analytics/click-analytics/?language=php#identifying-the-query-result-position].
     * @param objectIDs the object(s)' `objectID`.
     * @param positions the clicked object(s)' position(s).
     * @param timestamp the time at which the click happened. Defaults to current time.
     */
    @JvmOverloads
    fun clickedAfterSearch(
        eventName: String,
        queryId: String,
        objectIDs: EventObjects.IDs,
        positions: List<Int>,
        timestamp: Long = System.currentTimeMillis()
    ) = clicked(Event.Click(
        eventName = eventName,
        userToken = userTokenOrThrow(),
        timestamp = timestamp,
        eventObjects = objectIDs,
        queryId = queryId,
        positions = positions
    ))

    /**
     * Tracks a Conversion event, unrelated to a specific search query.
     *
     * @param eventName the event's name, **must not be empty**.
     * @param timestamp the time at which the conversion happened. Defaults to current time.
     * @param filters the converted filter(s).
     */
    @JvmOverloads
    fun converted(
        eventName: String,
        filters: EventObjects.Filters,
        timestamp: Long = System.currentTimeMillis()
    ) = converted(
        Event.Conversion(
            eventName = eventName,
            userToken = userTokenOrThrow(),
            timestamp = timestamp,
            eventObjects = filters
        )
    )

    /**
     * Tracks a Conversion event, unrelated to a specific search query.
     *
     * @param eventName the event's name, **must not be empty**.
     * @param objectIDs the object(s)' `objectID`.
     * @param timestamp the time at which the conversion happened. Defaults to current time.
     */
    @JvmOverloads
    fun converted(
        eventName: String,
        objectIDs: EventObjects.IDs,
        timestamp: Long = System.currentTimeMillis()
    ) = converted(
        Event.Conversion(
            eventName = eventName,
            userToken = userTokenOrThrow(),
            timestamp = timestamp,
            eventObjects = objectIDs
        )
    )

    /**
     * Tracks a Conversion event after a search has been done.
     *
     * @param eventName the event's name, **must not be empty**.
     * @param queryId the related [query's identifier][https://www.algolia.com/doc/guides/insights-and-analytics/click-analytics/?language=php#identifying-the-query-result-position].
     * @param objectIDs the object(s)' `objectID`.
     * @param timestamp the time at which the conversion happened. Defaults to current time.
     */
    @JvmOverloads
    fun convertedAfterSearch(
        eventName: String,
        queryId: String,
        objectIDs: EventObjects.IDs,
        timestamp: Long = System.currentTimeMillis()
    ) = converted(
        Event.Conversion(
            eventName = eventName,
            userToken = userTokenOrThrow(),
            timestamp = timestamp,
            eventObjects = objectIDs,
            queryId = queryId
        )
    )

    /**
     * Tracks a View event constructed manually.
     */
    fun viewed(event: Event.View) = track(event)

    /**
     * Tracks a Click event constructed manually.
     */
    fun clicked(event: Event.Click) = track(event)

    /**
     * Tracks a Conversion event, constructed manually.
     */
    fun converted(event: Event.Conversion) = track(event)

    /**
     * Method for tracking an event.
     * For a complete description of events see our [documentation][https://www.algolia.com/doc/rest-api/insights/?language=android#push-events].
     * @param [event] An [Event] that you want to track.
     */
    fun track(event: Event) {
        track(ConverterEventToEventInternal.convert(event to indexName))
    }

    private fun track(event: EventInternal) {
        if (enabled) {
            database.append(event)
            if (database.count() >= minBatchSize) {
                eventUploader.startOneTimeUpload()
            }
        }
    }

    // endregion

    override fun toString(): String {
        return "Insights(indexName='$indexName', webService=$webService)"
    }

    companion object {

        internal val insightsMap = mutableMapOf<String, Insights>()

        /**
         * Register your index with a given appId and apiKey.
         * @param context A [Context].
         * @param appId The given app id for which you want to track the events.
         * @param apiKey The API Key for your `appId`.
         * @param indexName The index that is being tracked.
         * @param configuration A [Configuration] class.
         * @return An [Insights] instance.
         */
        @JvmStatic
        fun register(
            context: Context,
            appId: String,
            apiKey: String,
            indexName: String,
            configuration: Configuration = Configuration(5000, 5000)
        ): Insights {
            val eventUploader = EventUploaderAndroidJob(context)
            val database = DatabaseSharedPreferences(context, indexName)
            val webService = WebServiceHttp(
                appId = appId,
                apiKey = apiKey,
                environment = WebServiceHttp.Environment.Prod,
                connectTimeoutInMilliseconds = configuration.connectTimeoutInMilliseconds,
                readTimeoutInMilliseconds = configuration.readTimeoutInMilliseconds
            )
            val insights = Insights(indexName, eventUploader, database, webService)
            insights.userToken = configuration.defaultUserToken

            val previousInsights = insightsMap.put(indexName, insights)
            previousInsights?.let {
                System.out.println("Registering new Insights for indexName $indexName. Previous instance: $insights")
            }
            shared = insights
            return insights
        }

        /**
         * Access an already registered `Insights` without having to pass the `apiKey` and `appId`.
         *
         * If the index was not register before, it will throw an [InsightsException.IndexNotRegistered] exception.
         * @param indexName The index that is being tracked.
         * @return An [Insights] instance.
         * @throws InsightsException.IndexNotRegistered if no index was registered as [indexName] before.
         */
        @JvmStatic
        fun shared(indexName: String): Insights {
            return insightsMap[indexName]
                ?: throw InsightsException.IndexNotRegistered()
        }

        /**
         * Access the latest registered `Insights` instance, if any.
         */
        @JvmStatic
        var shared: Insights? = null
            @JvmName("shared")
            get() = if (field != null) field else throw InsightsException.IndexNotRegistered()
    }
}
