package com.algolia.instantsearch.insights

import android.content.Context
import com.algolia.instantsearch.insights.database.Database
import com.algolia.instantsearch.insights.database.DatabaseSharedPreferences
import com.algolia.instantsearch.insights.event.Event
import com.algolia.instantsearch.insights.event.EventUploader
import com.algolia.instantsearch.insights.event.EventUploaderAndroidJob
import com.algolia.instantsearch.insights.webservice.WebService
import com.algolia.instantsearch.insights.webservice.WebServiceHttp

/**
 * Main class used for interacting with the InstantSearch Insights library.
 * In order to send insights, you first need to register an APP ID and API key for a given Index.
 * Once registered, you can simply call `Insights.shared(index:  String)` to send your events.
 * Example:
 * ```
 *   val indexName = "myAwesomeIndex"
 *   Insights.register(
 *      context = context,
 *      appId = "APPID",
 *      apiKey = "APIKEY",
 *      indexName = indexName
 *   )
 *   val clickData = mapOf(
 *      "eventName": "My super event",
 *      "queryID": "6de2f7eaa537fa93d8f8f05b927953b1",
 *      "position": 1,
 *      "objectID": "54675051",
 *      "indexName": indexName,
 *      "timestamp": Date.timeIntervalBetween1970AndReferenceDate
 *   )
 *   Insights.shared(index = indexName).click(params = data)
 * ```
 */
class Insights internal constructor(
    private val indexName: String,
    private val eventUploader: EventUploader,
    internal val database: Database,
    internal val webService: WebService
) {

    /**
     * Insights configuration.
     * @param connectTimeoutInMilliseconds Maximum amount of time in milliseconds before a connect timeout.
     * @param readTimeoutInMilliseconds Maximum amount of time in milliseconds before a read timeout.
     */
    class Configuration(
        val connectTimeoutInMilliseconds: Int,
        val readTimeoutInMilliseconds: Int
    )

    /**
     * Change this variable to `true` or `false` to enable or disable logging.
     * Use a filter on tag `Algolia Insights` to see all logs generated by the Insights library.
     */
    var loggingEnabled: Boolean = false
        set(value) {
            field = value
            InsightsLogger.enabled[indexName] = value
        }

    init {
        eventUploader.startPeriodicUpload()
    }

    /**
     * Method for tracking a click event.
     * For a complete list of mandatory fields see our [documentation][https://www.algolia.com/doc/rest-api/analytics/#post-click-event].
     * @param [params] A map of [Any] containing data points that you want to track.
     */
    fun click(params: Map<String, Any>) {
        process(Event.Click(params))
    }

    /**
     * Method for tracking a view event.
     * For a complete list of mandatory fields see our [documentation][https://www.algolia.com/doc/rest-api/analytics/#post-view-event].
     * @param [params] A map of [Any] containing data points that you want to track.
     */
    internal fun view(params: Map<String, Any>) {
        process(Event.View(params))
    }

    /**
     * Method for tracking a conversion event.
     * For a complete list of mandatory fields see our [documentation][https://www.algolia.com/doc/rest-api/analytics/#post-conversion-event].
     * @param [params] A map of [Any] containing data points that you want to track.
     */
    fun conversion(params: Map<String, Any>) {
        process(Event.Conversion(params))
    }

    private fun process(event: Event) {
        database.append(event)
        eventUploader.startOneTimeUpload()
    }

    companion object {

        internal val insightsMap = mutableMapOf<String, Insights>()

        /**
         * Register your index with a given appId and apiKey.
         * @param context A [Context].
         * @param appId The given app id for which you want to track the events.
         * @param apiKey The API Key for your `appId`.
         * @param indexName The index that is being tracked.
         * @param configuration A [Configuration] class.
         * @return An [Insights] instance.
         */
        @JvmStatic
        fun register(
            context: Context,
            appId: String,
            apiKey: String,
            indexName: String,
            configuration: Configuration = Configuration(5000, 5000)
        ): Insights {
            val eventUploader = EventUploaderAndroidJob(context)
            val database = DatabaseSharedPreferences(context, indexName)
            val webService = WebServiceHttp(
                appId = appId,
                apiKey = apiKey,
                environment = WebServiceHttp.Environment.Prod,
                connectTimeoutInMilliseconds = configuration.connectTimeoutInMilliseconds,
                readTimeoutInMilliseconds = configuration.readTimeoutInMilliseconds
            )
            val insights = Insights(indexName, eventUploader, database, webService)

            insightsMap[indexName] = insights
            return insights
        }

        /**
         * Access an already registered `Insights` without having to pass the `apiKey` and `appId`.
         * If the index was not register before, it will throw an [InstantSearchInsightsException.IndexNotRegistered] exception.
         * @param indexName The index that is being tracked.
         * @return An [Insights] instance.
         */
        @JvmStatic
        fun shared(indexName: String): Insights {
            return insightsMap[indexName]
                ?: throw InstantSearchInsightsException.IndexNotRegistered()
        }
    }
}
