package com.polestar.sensors;

import android.content.Context;
import android.content.ContextWrapper;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.os.Looper;
import android.view.Surface;
import android.view.WindowManager;

import com.polestar.helpers.Log;
import com.polestar.models.ISensorListener;
import com.polestar.models.MemsData;
import com.polestar.models.MemsMeasurement;
import com.polestar.models.SensorNao;
import com.polestar.naosdk.api.ISensorObserver;
import com.polestar.naosdk.api.TSENSORPOWERMODE;
import com.polestar.naosdk.api.TSENSORTYPE;

import java.util.Iterator;
import java.util.List;

public class MemsSensor  extends SensorNao implements SensorEventListener{

	private static final String MEMS_SENSOR_THREAD_NAME = "MS_Thread"; // name of the thread MEMS
	/** Default for the period of scan in s */
	public static final double MEMS_SENSOR_DEFAULT_PERIOD_SECONDS = 0.015;
	public static final long MEMS_SENSOR_DEFAULT_TIMESTAMP_RESOLUTION_NANOS = 1;
	public static final double MEMS_SENSOR_DEFAULT_PERIOD_ACCUMULATOR_SECONDS  = 1.0;
	public static final int MEMS_SENSOR_DELAY = SensorManager.SENSOR_DELAY_GAME;
	/** The object to which this sensor sends its data when available */
	private ISensorListener mObserver;

	/** The temp data to send to the receiver */
	private MemsData memsData;
	private MemsMeasurement memsMeasurement; // accumulated memsData
	private float[] values; //temp values for 3 axis vector

	private float[] gravity;
	private float[] orientation;
	private float[] rotationMatrix;
	private float[] inclinationMatrix;

	/** The timers to collect and push mems data to the receiver */
	private long lastCollectedDataTimestamp;
	private long lastPushedDataTimestamp;
	private long firstEventTimestamp;
	private long firstEventSystemTimeMillis;

	/** use Window Manager to get the current view screen rotation */
	private int screenRotation;

    /** The android sensors */
    private SensorManager sensorManager;
    private List<Sensor> sensors;

	/** Broadcast Receiver which listen if the screen is off **/
	static private String [] phonesWithNoHeadingAccuracy = {"ST"};

	/** Indicates if it is the first event */
	private boolean firstSensorEvent;

	/** force gyro to turn on when order provenance is from logger*/
	private boolean turnOnGyro = false;



	/**
	 * Constructor
	 * @param caller, the context
	 */
	public MemsSensor(ContextWrapper caller, ISensorObserver observer) {
		super();
		setSensorObserver(observer);
		mThread = new Thread(this);
		mThread.setName(MEMS_SENSOR_THREAD_NAME);
		mThread.start();

		if (caller == null) {
			mStatus = STATUS_ERROR;
		}else{
			this.mCtxtWrap = caller;
			this.mStatus = STATUS_READY;
		}
	}

	/**
	 *	thread run, MemsSensor is a thread
	 */
	public void run() {
		Looper.prepare();
		myLooper = Looper.myLooper();
		this.mOrder = ORDER_TURN_OFF;
		Looper.loop();
	}

	/**
	 * register to listen to MEMS Android sensors and start sensing
	 * @return
	 */
	private boolean initMemsSensors(){
		// the data
		memsData = new MemsData();
		memsMeasurement = new MemsMeasurement();
		values = new float[3];

		gravity = new float[3];

		orientation = new float[3];
		rotationMatrix = new float[9];
		inclinationMatrix = new float[9];

		// the timestamps
		lastCollectedDataTimestamp = 0;
		lastPushedDataTimestamp = 0;

		//Indicates if it is the first sensor event
		this.firstSensorEvent = true;

		// the sensors
		sensorManager = (SensorManager)mCtxtWrap.getSystemService(Context.SENSOR_SERVICE);
		sensors = sensorManager.getSensorList(Sensor.TYPE_ALL); // get all available sensor
		// and register for all of them
		boolean turnOn ;
		for (Iterator<Sensor> iterator = sensors.iterator(); iterator.hasNext();) {
			Sensor sensor = (Sensor) iterator.next();
			turnOn = false ;

			switch (sensor.getType()) {
				case Sensor.TYPE_ACCELEROMETER:
				case Sensor.TYPE_GRAVITY:
				case Sensor.TYPE_ROTATION_VECTOR:
				case Sensor.TYPE_LINEAR_ACCELERATION:
					turnOn = true;
					break;
				case Sensor.TYPE_GYROSCOPE:
					// Activate gyroscope only for the logger
					if (this.mOrderProvenance == ORDER_PROVENANCE_LOGGER) {
						turnOnGyro = true;
					}
					turnOn = turnOnGyro;
                    break;
				case Sensor.TYPE_MAGNETIC_FIELD: 	//  it will be launched by algo
					// (to prevent asking calibration too early)
				default:
					turnOn = false ;
					break;
			}
			if(turnOn)
			{
				sensorManager.registerListener(this, sensor, MEMS_SENSOR_DELAY); // rate in microseconds
			}
		}

		return true;
	}

	/**
	 * regsiter sensor
	 * @param sensor
	 */
	public void registerSensor(Sensor sensor) {
		sensorManager.registerListener(this, sensor, MEMS_SENSOR_DELAY); // rate in microseconds
	}

	/**
	 * unregsiter sensor
	 * @param sensor
	 */
	public void unregisterSensor(Sensor sensor) {
		sensorManager.unregisterListener(this, sensor); // rate in microseconds
	}


	/**
	 *  clean-up MEMS sensors and stop sensing
	 */
	synchronized private void finalizeMemsSensors(){
		if(sensorManager!=null){
			sensorManager.unregisterListener(this);
		}
	}

	/**
	 *  collect MEMS measurement from sensors into the accumulator
	 */

	synchronized private void collectMemsData(SensorEvent event){
		if(memsData == null || memsMeasurement==null) {
			return;
		}

		// if the timestamp is new, push previously collected data for all sensors
		if (event.timestamp - lastCollectedDataTimestamp > MEMS_SENSOR_DEFAULT_TIMESTAMP_RESOLUTION_NANOS){

			//Log.alwaysWarn(this.getClass().getName(), event.sensor.getType() + " MEMS DATA COLLECTION  ------------------------------------------------ ");
			if (lastCollectedDataTimestamp > 0) {
				MemsData memsDataCopy = new MemsData(memsData);
				memsMeasurement.addMemsData(memsDataCopy);
				memsData.rotationRate[0]=0;
				memsData.rotationRate[1]=0;
				memsData.rotationRate[2]=0;
			}
			lastCollectedDataTimestamp = event.timestamp;
		}

		//then update mems Data with the current sensor
		long eventTimeMillis = firstEventSystemTimeMillis + (event.timestamp - firstEventTimestamp) / 1000000L ;
		memsData.timestamp = eventTimeMillis;

		if (this.mStatus == STATUS_RUNNING){

			//correction of 3 axis values depending on screen orientation
			this.screenRotation = ((WindowManager) this.mCtxtWrap.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay().getRotation();

			values = event.values.clone();

			//update local gravity before screen rotation compensation and normalisation
			// (local gravity is used for magHeading computation)
			gravity = values.clone();

			if (this.screenRotation == Surface.ROTATION_270) //Case landscape right
			{
				values[0] = event.values[1];
				values[1] = -event.values[0];
			}
			else if (this.screenRotation == Surface.ROTATION_90) //Case landscape left
			{
				values[0] = -event.values[1];
				values[1] = event.values[0];
			}
			else if (this.screenRotation == Surface.ROTATION_180) //Case portrait upsideDown
			{
				values[0] = -event.values[0];
				values[1] = -event.values[1];
			}

			//apply gravity normalisation (to be compliant with Pole Star sensors format - same as iOS)
			if (event.sensor.getType() == Sensor.TYPE_ACCELEROMETER ||
					event.sensor.getType() == Sensor.TYPE_GRAVITY ||
					event.sensor.getType() == Sensor.TYPE_LINEAR_ACCELERATION) {
				for (int i=0; i<3; i++) {
					values[i] = - values[i] / SensorManager.STANDARD_GRAVITY;
				}
			}

			// override the current MemsData
			switch (event.sensor.getType()) {
				case Sensor.TYPE_ACCELEROMETER:
					memsData.acceleration = values.clone();
					memsMeasurement.setAccelAvailable(true);

				case Sensor.TYPE_GRAVITY:
					memsData.gravity = values.clone();
					memsMeasurement.setGravityAvailable(true);

				case Sensor.TYPE_LINEAR_ACCELERATION:
					memsData.userAcceleration = values.clone();
					break;

				case Sensor.TYPE_MAGNETIC_FIELD:
					memsData.magField = values.clone();
					memsMeasurement.setMagFiledAvailable(true);

					memsData.headingAccuracy = event.accuracy;

					SensorManager.getRotationMatrix(rotationMatrix, inclinationMatrix, gravity, event.values);
					SensorManager.getOrientation(rotationMatrix, orientation);
					memsData.magHeading = (float) Math.toDegrees(orientation[0]) + this.screenRotation * 90;
					break;

				case Sensor.TYPE_GYROSCOPE:
					memsData.rotationRate = values.clone();
					memsMeasurement.setGyroAvailable(true);
					break;

				case Sensor.TYPE_ROTATION_VECTOR:
					memsData.rotation = values.clone();
					memsMeasurement.setRotationAvailable(true);

					SensorManager.getRotationMatrixFromVector(rotationMatrix, event.values);
					SensorManager.getOrientation(rotationMatrix, orientation);
					memsData.trueHeading = (float) Math.toDegrees(orientation[0]) + this.screenRotation * 90;
					break;

				default:
					break;
			}
		}
	}


	/**
	 * push the accumulated MEMS data to the recipient
	 */
	private void pushMemsData(){
		if (this.mStatus == STATUS_RUNNING){
			if (mObserver != null && memsMeasurement!=null){
				mObserver.onNewMeasurement(memsMeasurement);
			}

			if(mSensorObserver != null) {
				mSensorObserver.notifyOfNewData(TSENSORTYPE.MEMS, memsMeasurement.toByteArray());
			}

			memsMeasurement = new MemsMeasurement();

		}
	}


	/**
	 * initMemsSensors
	 * initTimers
	 * state READY -> RUNNING or ERROR
	 */
	@Override
	public boolean startSensor() {
		if (this.mStatus != STATUS_READY) {
			Log.alwaysWarn(this.getClass().getName(), "Cannot start MEMS sensor: previous state = " + this.mStatus);
			return false;
		}
		if (!initMemsSensors()) {
			this.mStatus = STATUS_ERROR;
			Log.alwaysWarn(this.getClass().getName(), "Cannot start MEMS sensor: not available = " + this.mStatus);
			return false;
		}
		this.mStatus = STATUS_RUNNING;
		this.mOrder = ORDER_TURN_ON;
		return true;
	}

	/**
	 * Function called by sensor owner for the sensor to properly stop before it completely ceases functionning.
	 * The sensor cannot be used any more after this call. This effectively exits its thread main loop.
	 * Has no effect if status is not RUNNING
	 * cancelTimers
	 * finalizeMemsSensors
	 * state RUNNING -> READY
	 */
	@Override
	public void stopSensor() {
		if (this.mStatus != STATUS_RUNNING) {
			Log.restricted(this.getClass().getName(), "Cannot stop MEMS sensor: previous state = " + this.mStatus);
			return;
		}
		finalizeMemsSensors();

		this.mStatus = STATUS_READY;
		this.mOrder = ORDER_TURN_OFF;
	}

	@Override
	public void start() {
		Log.alwaysWarn(this.getClass().getName(),"Starting Mems sensors from native .....");
		switchSensor(SensorNao.ORDER_TURN_ON, SensorNao.ORDER_PROVENANCE_SERVICE);
	}

	@Override
	public boolean isRunning() {
		return (mStatus == STATUS_RUNNING);
	}

	@Override
	public void stop() {
		Log.alwaysWarn(this.getClass().getName(),"Stoping Mems sensors from native .....");
		switchSensor(SensorNao.ORDER_TURN_OFF, SensorNao.ORDER_PROVENANCE_SERVICE);
	}

	/**
	 * stops and restarts the sensor thread
	 * has no effect if state is not RUNNING.
	 *
	 * WARNING : this method may take a few hundred milliseconds and even a few seconds, since it blocks until the
	 * sensor thread ('main' method) ends.
	 **/
	@Override
	public void reset() {
		if (this.mStatus == STATUS_RUNNING && this.mOrder == ORDER_TURN_ON) {
			switchSensor(ORDER_TURN_OFF,ORDER_PROVENANCE_SERVICE);
			switchSensor(ORDER_TURN_ON,ORDER_PROVENANCE_SERVICE);
		}

	}

	@Override
	public void setPowerMode(TSENSORPOWERMODE mode) {
		//Not implemented
	}

	@Override
	public void registerOutputInterface(ISensorListener targetInterface) {
		this.mObserver = targetInterface;
	}

	@Override
	public boolean isOn() {
		return (this.mStatus == STATUS_RUNNING);
	}

	@Override
	public boolean hasFix() {
		return false;
	}

	@Override
	public boolean isActive() {
		return isHardwareCompatible();
	}

	/**
	 * get if sensor is HW compatible
	 */
	public boolean isHardwareCompatible() {

		return (mCtxtWrap.getSystemService(Context.SENSOR_SERVICE) != null);
	}

	/**
	 * destructor : cleanly stops the service.
	 */
	protected void finalize() {
		//stopSensor();
	}

	/**
	 * method from SensorEventListener 
	 */
	public void onAccuracyChanged(Sensor sensor, int accuracy) {
		// TODO Auto-generated method stub
	}

	/**
	 * method from SensorEventListener 
	 */
	public void onSensorChanged(SensorEvent event) {
		if (event != null){

			if(this.firstSensorEvent)
			{
				lastPushedDataTimestamp = event.timestamp ;
				firstEventTimestamp = event.timestamp ;
				firstEventSystemTimeMillis = System.currentTimeMillis() ;
				this.firstSensorEvent = false;
			}

			collectMemsData(event);

			if (event.timestamp - lastPushedDataTimestamp > MEMS_SENSOR_DEFAULT_PERIOD_ACCUMULATOR_SECONDS*Math.pow(10,9)){
				//Log.alwaysWarn(this.getClass().getName(), "============send " + (int)((event.timestamp - lastPushedDataTimestamp)*Math.pow(10,-6)) + " ms " + memsMeasurement.getNbMeas() + " meas");//DBG
				lastPushedDataTimestamp = event.timestamp;
				pushMemsData();
			}
		}
	}

	public static void setPhonesWithNoHeadingAccuracy(
			String [] phonesWithNoHeadingAccuracy) {
		MemsSensor.phonesWithNoHeadingAccuracy = phonesWithNoHeadingAccuracy;
	}

	public static String [] getPhonesWithNoHeadingAccuracy() {
		return phonesWithNoHeadingAccuracy;
	}

	@Override
	public boolean setStateAsBefore() {
		// TODO Auto-generated method stub
		return false;
	}


}

